/***************************************************************************
    copyright            : (C) 2013 by Tsuda Kageyu
    email                : tsuda.kageyu@gmail.com
 ***************************************************************************/

/***************************************************************************
 *   This library is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License version   *
 *   2.1 as published by the Free Software Foundation.                     *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301  USA                                                       *
 *                                                                         *
 *   Alternatively, this file is available under the Mozilla Public        *
 *   License Version 1.1.  You may obtain a copy of the License at         *
 *   http://www.mozilla.org/MPL/                                           *
 ***************************************************************************/

#ifndef TAGLIB_REFCOUNTER_H
#define TAGLIB_REFCOUNTER_H

#include "taglib_export.h"
#include "taglib.h"

#if defined(HAVE_MAC_ATOMIC)
#  include <libkern/OSAtomic.h>
#elif defined(HAVE_WIN_ATOMIC)
#  define NOMINMAX
#  include <windows.h>
#elif defined(HAVE_IA64_ATOMIC)
#  include <ia64intrin.h>
#endif

#ifndef DO_NOT_DOCUMENT // Tell Doxygen to skip this class.
/*!
  * \internal
  * This is just used as a base class for shared classes in TagLib.
  *
  * \warning This <b>is not</b> part of the TagLib public API!
  */
namespace TagLib
{

  class TAGLIB_EXPORT RefCounter
  {
  public:
    RefCounter();
    virtual ~RefCounter();

    void ref();
    bool deref();
    int count() const;

  private:
    class RefCounterPrivate;
    RefCounterPrivate *d;
  };

  // BIC this old class is needed by tlist.tcc and tmap.tcc
  class RefCounterOld
  {
  public:
    RefCounterOld() : refCount(1) {}

#if defined(HAVE_MAC_ATOMIC)
    void ref() { OSAtomicIncrement32Barrier(const_cast<int32_t*>(&refCount)); }
    bool deref() { return ! OSAtomicDecrement32Barrier(const_cast<int32_t*>(&refCount)); }
    int32_t count() { return refCount; }
  private:
    volatile int32_t refCount;
#elif defined(HAVE_WIN_ATOMIC)
    void ref() { InterlockedIncrement(&refCount); }
    bool deref() { return ! InterlockedDecrement(&refCount); }
    long count() { return refCount; }
  private:
    volatile long refCount;
#elif defined(HAVE_GCC_ATOMIC)
    void ref() { __sync_add_and_fetch(&refCount, 1); }
    bool deref() { return ! __sync_sub_and_fetch(&refCount, 1); }
    int count() { return refCount; }
  private:
    volatile int refCount;
#else
    void ref() { refCount++; }
    bool deref() { return ! --refCount; }
    int count() { return refCount; }
  private:
    uint refCount;
#endif
  };

}

#endif // DO_NOT_DOCUMENT
#endif

