/*
 *  Exercise all code paths required by the current fastint implementation.
 *
 *  The fastint number model is transparent to user code.  Numbers are
 *  represented either as IEEE doubles or 48-bit signed integers.  See
 *  internal fastint documentation for details.
 *
 *  Expect strings are based on V8 and Rhino - careful with the following
 *  Rhino bug:
 *
 *      $ node
 *      > 9223372036854776000 >>> -18446744073709552000
 *      0
 *
 *      $ rhino
 *      Rhino 1.7 release 4 2013 08 27
 *      js> 9223372036854776000 >>> -18446744073709552000
 *      4294967295
 */

/*@include util-object.js@*/

/*---
{
    "custom": true,
    "specialoptions": "requires DUK_USE_FASTINT"
}
---*/

var MIN_FASTINT = -140737488355328;
var MIN_FASTINT_PLUS1 = -140737488355327;
var MAX_FASTINT = 140737488355327;
var MAX_FASTINT_EVEN = MAX_FASTINT - 1;
var MAX_FASTINT_HALF = MAX_FASTINT_EVEN / 2;
var POS_ZERO = 0;
var NEG_ZERO = -1 / (1 / 0);

var TEST_VALUES = [
    -0xffffffffffffffff,
    -0x8000000000000000,
    -0x7fffffffffffffff,
    -0x0000800000000001,
    -0x0000800000000000,  // smallest fastint
    +0x00007fffffffffff,  // largest fastint
    +0x0000800000000000,
    +0x7fffffffffffffff,
    +0x8000000000000000,
    +0xffffffffffffffff,

    -0xffffffff,
    -0x80000001,
    -0x80000000,
    -0x7fffffff,
    +0x7fffffff,
    +0x80000000,
    +0x80000001,
    +0xffffffff,

    -3, -2, -1, NEG_ZERO, POS_ZERO, 1, 2, 3
];

function printFixed() {
    var res = [];
    var i, t;

    for (i = 0; i < arguments.length; i++) {
        t = arguments[i];
        if (t === 0) {
            if (1 / t > 0) { res.push('+0'); }
            else { res.push('-0'); }
        } else {
            res.push(String(t));
        }
    }

    print(res.join(' '));
}

function printFastint(v) {
    var prefix = '';
    var isfast = '';

    if (v === 0) {
        if (1 / v > 0) { prefix = '+'; }
        else { prefix = '-'; }
    }

    // The internal type tag depends on the duk_tval unpacked/packed layout.
    // XXX: add to util.
    if (typeof Duktape !== 'object') {
        isfast = ' NOT-DUKTAPE';
    } else if (getValueInternalTag(true) >= 0xfff0) {
        // packed duk_tval
        if (getValueInternalTag(v) === 0xfff1) {
            isfast = ' fastint';
        }
    } else if (getValueInternalTag(true) === 4) {
        // non-packed duk_tval
        if (getValueInternalTag(v) === 1) {
            isfast = ' fastint';
        }
    } else {
        isfast = ' CANNOT-DETERMINE';
    }

    print(prefix + v, v.toString(16) + isfast);
}

/*===
-140737488355328 -800000000000 fastint
-140737488355327 -7fffffffffff fastint
140737488355327 7fffffffffff fastint
140737488355326 7ffffffffffe fastint
70368744177663 3fffffffffff fastint
+0 0 fastint
-0 0
===*/

printFastint(MIN_FASTINT);
printFastint(MIN_FASTINT_PLUS1);
printFastint(MAX_FASTINT);
printFastint(MAX_FASTINT_EVEN);
printFastint(MAX_FASTINT_HALF);
printFastint(POS_ZERO);
printFastint(NEG_ZERO);

/*===
binary arith in range
-140737488355327 -7fffffffffff fastint
140737488355327 7fffffffffff fastint
140737488355326 7ffffffffffe fastint
-140737488355328 -800000000000 fastint
140737488289792 7fffffff0000 fastint
-140737488289792 -7fffffff0000 fastint
70368744177663 3fffffffffff fastint
-70368744177664 -400000000000 fastint
4 4 fastint
1 1 fastint
binary arith out of range
140737488355328 800000000000
-140737488355329 -800000000001
140739635773439 80007ffeffff
-140739635773439 -80007ffeffff
70368744177663.5 3fffffffffff.8
33.333333333333336 21.555555555556
Infinity Infinity
1 1
NaN NaN
===*/

function binaryArithmeticTest() {
    var x, y, z;

    print('binary arith in range');

    x = MIN_FASTINT; y = 1; z = x + y; printFastint(z);
    x = MAX_FASTINT - 1; y = 1; z = x + y; printFastint(z);

    x = MAX_FASTINT; y = 1; z = x - y; printFastint(z);
    x = MIN_FASTINT + 1; y = 1; z = x - y; printFastint(z);

    // multiplication is restricted to both inputs having signed 32-bit values
    // (and range in 48-bit signed range of course)
    x = 0x7fffffff; y = 0x10000; z = x * y; printFastint(z);
    x = -0x7fffffff; y = 0x10000; z = x * y; printFastint(z);

    // division is restricted to non-fractional results
    x = MAX_FASTINT_EVEN; y = 2; z = x / y; printFastint(z);
    x = MIN_FASTINT; y = 2; z = x / y; printFastint(z);
    x = 100; y = 25; z = x / y; printFastint(z);

    // modulus is restricted to both arguments being >= 1
    x = MAX_FASTINT; y = 3; z = x % y; printFastint(z);

    print('binary arith out of range');  // out of range or fractional

    x = MAX_FASTINT; y = 1; z = x + y; printFastint(z);

    x = MIN_FASTINT; y = 1; z = x - y; printFastint(z);

    x = 0x7fffffff; y = 0x10001; z = x * y; printFastint(z);
    x = -0x7fffffff; y = 0x10001; z = x * y; printFastint(z);

    x = MAX_FASTINT; y = 2; z = x / y; printFastint(z);  // fractional
    x = 100; y = 3; z = x / y; printFastint(z);
    x = 100; y = 0; z = x / y; printFastint(z);

    x = MAX_FASTINT; y = -3; z = x % y; printFastint(z);
    x = MAX_FASTINT; y = 0; z = x % y; printFastint(z);
}

/*===
binary brute force
-18446744073709552000 -18446744073709552000 -36893488147419103000 +0 3.402823669209385e+38 1 -0
-18446744073709552000 -9223372036854776000 -27670116110564327000 -9223372036854776000 1.7014118346046923e+38 2 -0
-18446744073709552000 -9223372036854776000 -27670116110564327000 -9223372036854776000 1.7014118346046923e+38 2 -0
-18446744073709552000 -140737488355329 -18446884811197907000 -18446603336221196000 2.596148429267432e+33 131071.99999999907 -140737488224257
-18446744073709552000 -140737488355328 -18446884811197907000 -18446603336221196000 2.596148429267414e+33 131072 -0
-18446744073709552000 140737488355327 -18446603336221196000 -18446884811197907000 -2.5961484292673954e+33 -131072.00000000093 -131072
-18446744073709552000 140737488355328 -18446603336221196000 -18446884811197907000 -2.596148429267414e+33 -131072 -0
-18446744073709552000 9223372036854776000 -9223372036854776000 -27670116110564327000 -1.7014118346046923e+38 -2 -0
-18446744073709552000 9223372036854776000 -9223372036854776000 -27670116110564327000 -1.7014118346046923e+38 -2 -0
-18446744073709552000 18446744073709552000 +0 -36893488147419103000 -3.402823669209385e+38 -1 -0
-18446744073709552000 -4294967295 -18446744078004520000 -18446744069414584000 7.922816249581759e+28 4294967297 -1
-18446744073709552000 -2147483649 -18446744075857035000 -18446744071562068000 3.961408127557891e+28 8589934588 -4
-18446744073709552000 -2147483648 -18446744075857035000 -18446744071562068000 3.961408125713217e+28 8589934592 -0
-18446744073709552000 -2147483647 -18446744075857035000 -18446744071562068000 3.9614081238685425e+28 8589934596 -4
-18446744073709552000 2147483647 -18446744071562068000 -18446744075857035000 -3.9614081238685425e+28 -8589934596 -4
-18446744073709552000 2147483648 -18446744071562068000 -18446744075857035000 -3.961408125713217e+28 -8589934592 -0
-18446744073709552000 2147483649 -18446744071562068000 -18446744075857035000 -3.961408127557891e+28 -8589934588 -4
-18446744073709552000 4294967295 -18446744069414584000 -18446744078004520000 -7.922816249581759e+28 -4294967297 -1
-18446744073709552000 -3 -18446744073709552000 -18446744073709552000 55340232221128655000 6148914691236517000 -1
-18446744073709552000 -2 -18446744073709552000 -18446744073709552000 36893488147419103000 9223372036854776000 -0
-18446744073709552000 -1 -18446744073709552000 -18446744073709552000 18446744073709552000 18446744073709552000 -0
-18446744073709552000 -0 -18446744073709552000 -18446744073709552000 +0 Infinity NaN
-18446744073709552000 +0 -18446744073709552000 -18446744073709552000 -0 -Infinity NaN
-18446744073709552000 1 -18446744073709552000 -18446744073709552000 -18446744073709552000 -18446744073709552000 -0
-18446744073709552000 2 -18446744073709552000 -18446744073709552000 -36893488147419103000 -9223372036854776000 -0
-18446744073709552000 3 -18446744073709552000 -18446744073709552000 -55340232221128655000 -6148914691236517000 -1
-9223372036854776000 -18446744073709552000 -27670116110564327000 9223372036854776000 1.7014118346046923e+38 0.5 -9223372036854776000
-9223372036854776000 -9223372036854776000 -18446744073709552000 +0 8.507059173023462e+37 1 -0
-9223372036854776000 -9223372036854776000 -18446744073709552000 +0 8.507059173023462e+37 1 -0
-9223372036854776000 -140737488355329 -9223512774343131000 -9223231299366420000 1.298074214633716e+33 65535.999999999534 -140737488289793
-9223372036854776000 -140737488355328 -9223512774343131000 -9223231299366420000 1.298074214633707e+33 65536 -0
-9223372036854776000 140737488355327 -9223231299366420000 -9223512774343131000 -1.2980742146336977e+33 -65536.00000000047 -65536
-9223372036854776000 140737488355328 -9223231299366420000 -9223512774343131000 -1.298074214633707e+33 -65536 -0
-9223372036854776000 9223372036854776000 +0 -18446744073709552000 -8.507059173023462e+37 -1 -0
-9223372036854776000 9223372036854776000 +0 -18446744073709552000 -8.507059173023462e+37 -1 -0
-9223372036854776000 18446744073709552000 9223372036854776000 -27670116110564327000 -1.7014118346046923e+38 -0.5 -9223372036854776000
-9223372036854776000 -4294967295 -9223372041149743000 -9223372032559809000 3.9614081247908797e+28 2147483648.5 -2147483648
-9223372036854776000 -2147483649 -9223372039002260000 -9223372034707292000 1.9807040637789456e+28 4294967294 -2
-9223372036854776000 -2147483648 -9223372039002260000 -9223372034707292000 1.9807040628566084e+28 4294967296 -0
-9223372036854776000 -2147483647 -9223372039002260000 -9223372034707292000 1.9807040619342712e+28 4294967298 -2
-9223372036854776000 2147483647 -9223372034707292000 -9223372039002260000 -1.9807040619342712e+28 -4294967298 -2
-9223372036854776000 2147483648 -9223372034707292000 -9223372039002260000 -1.9807040628566084e+28 -4294967296 -0
-9223372036854776000 2147483649 -9223372034707292000 -9223372039002260000 -1.9807040637789456e+28 -4294967294 -2
-9223372036854776000 4294967295 -9223372032559809000 -9223372041149743000 -3.9614081247908797e+28 -2147483648.5 -2147483648
-9223372036854776000 -3 -9223372036854776000 -9223372036854776000 27670116110564327000 3074457345618258400 -2
-9223372036854776000 -2 -9223372036854776000 -9223372036854776000 18446744073709552000 4611686018427388000 -0
-9223372036854776000 -1 -9223372036854776000 -9223372036854776000 9223372036854776000 9223372036854776000 -0
-9223372036854776000 -0 -9223372036854776000 -9223372036854776000 +0 Infinity NaN
-9223372036854776000 +0 -9223372036854776000 -9223372036854776000 -0 -Infinity NaN
-9223372036854776000 1 -9223372036854776000 -9223372036854776000 -9223372036854776000 -9223372036854776000 -0
-9223372036854776000 2 -9223372036854776000 -9223372036854776000 -18446744073709552000 -4611686018427388000 -0
-9223372036854776000 3 -9223372036854776000 -9223372036854776000 -27670116110564327000 -3074457345618258400 -2
-9223372036854776000 -18446744073709552000 -27670116110564327000 9223372036854776000 1.7014118346046923e+38 0.5 -9223372036854776000
-9223372036854776000 -9223372036854776000 -18446744073709552000 +0 8.507059173023462e+37 1 -0
-9223372036854776000 -9223372036854776000 -18446744073709552000 +0 8.507059173023462e+37 1 -0
-9223372036854776000 -140737488355329 -9223512774343131000 -9223231299366420000 1.298074214633716e+33 65535.999999999534 -140737488289793
-9223372036854776000 -140737488355328 -9223512774343131000 -9223231299366420000 1.298074214633707e+33 65536 -0
-9223372036854776000 140737488355327 -9223231299366420000 -9223512774343131000 -1.2980742146336977e+33 -65536.00000000047 -65536
-9223372036854776000 140737488355328 -9223231299366420000 -9223512774343131000 -1.298074214633707e+33 -65536 -0
-9223372036854776000 9223372036854776000 +0 -18446744073709552000 -8.507059173023462e+37 -1 -0
-9223372036854776000 9223372036854776000 +0 -18446744073709552000 -8.507059173023462e+37 -1 -0
-9223372036854776000 18446744073709552000 9223372036854776000 -27670116110564327000 -1.7014118346046923e+38 -0.5 -9223372036854776000
-9223372036854776000 -4294967295 -9223372041149743000 -9223372032559809000 3.9614081247908797e+28 2147483648.5 -2147483648
-9223372036854776000 -2147483649 -9223372039002260000 -9223372034707292000 1.9807040637789456e+28 4294967294 -2
-9223372036854776000 -2147483648 -9223372039002260000 -9223372034707292000 1.9807040628566084e+28 4294967296 -0
-9223372036854776000 -2147483647 -9223372039002260000 -9223372034707292000 1.9807040619342712e+28 4294967298 -2
-9223372036854776000 2147483647 -9223372034707292000 -9223372039002260000 -1.9807040619342712e+28 -4294967298 -2
-9223372036854776000 2147483648 -9223372034707292000 -9223372039002260000 -1.9807040628566084e+28 -4294967296 -0
-9223372036854776000 2147483649 -9223372034707292000 -9223372039002260000 -1.9807040637789456e+28 -4294967294 -2
-9223372036854776000 4294967295 -9223372032559809000 -9223372041149743000 -3.9614081247908797e+28 -2147483648.5 -2147483648
-9223372036854776000 -3 -9223372036854776000 -9223372036854776000 27670116110564327000 3074457345618258400 -2
-9223372036854776000 -2 -9223372036854776000 -9223372036854776000 18446744073709552000 4611686018427388000 -0
-9223372036854776000 -1 -9223372036854776000 -9223372036854776000 9223372036854776000 9223372036854776000 -0
-9223372036854776000 -0 -9223372036854776000 -9223372036854776000 +0 Infinity NaN
-9223372036854776000 +0 -9223372036854776000 -9223372036854776000 -0 -Infinity NaN
-9223372036854776000 1 -9223372036854776000 -9223372036854776000 -9223372036854776000 -9223372036854776000 -0
-9223372036854776000 2 -9223372036854776000 -9223372036854776000 -18446744073709552000 -4611686018427388000 -0
-9223372036854776000 3 -9223372036854776000 -9223372036854776000 -27670116110564327000 -3074457345618258400 -2
-140737488355329 -18446744073709552000 -18446884811197907000 18446603336221196000 2.596148429267432e+33 0.000007629394531250054 -140737488355329
-140737488355329 -9223372036854776000 -9223512774343131000 9223231299366420000 1.298074214633716e+33 0.00001525878906250011 -140737488355329
-140737488355329 -9223372036854776000 -9223512774343131000 9223231299366420000 1.298074214633716e+33 0.00001525878906250011 -140737488355329
-140737488355329 -140737488355329 -281474976710658 +0 1.9807040628566366e+28 1 -0
-140737488355329 -140737488355328 -281474976710657 -1 1.9807040628566225e+28 1.000000000000007 -1
-140737488355329 140737488355327 -2 -281474976710656 -1.9807040628566084e+28 -1.0000000000000142 -2
-140737488355329 140737488355328 -1 -281474976710657 -1.9807040628566225e+28 -1.000000000000007 -1
-140737488355329 9223372036854776000 9223231299366420000 -9223512774343131000 -1.298074214633716e+33 -0.00001525878906250011 -140737488355329
-140737488355329 9223372036854776000 9223231299366420000 -9223512774343131000 -1.298074214633716e+33 -0.00001525878906250011 -140737488355329
-140737488355329 18446744073709552000 18446603336221196000 -18446884811197907000 -2.596148429267432e+33 -0.000007629394531250054 -140737488355329
-140737488355329 -4294967295 -140741783322624 -140733193388034 6.044629096665814e+23 32768.00000762963 -32769
-140737488355329 -2147483649 -140739635838978 -140735340871680 3.022314550443969e+23 65535.99996948289 -2147418114
-140737488355329 -2147483648 -140739635838977 -140735340871681 3.0223145490365944e+23 65536.00000000047 -1
-140737488355329 -2147483647 -140739635838976 -140735340871682 3.0223145476292195e+23 65536.00003051804 -65537
-140737488355329 2147483647 -140735340871682 -140739635838976 -3.0223145476292195e+23 -65536.00003051804 -65537
-140737488355329 2147483648 -140735340871681 -140739635838977 -3.0223145490365944e+23 -65536.00000000047 -1
-140737488355329 2147483649 -140735340871680 -140739635838978 -3.022314550443969e+23 -65535.99996948289 -2147418114
-140737488355329 4294967295 -140733193388034 -140741783322624 -6.044629096665814e+23 -32768.00000762963 -32769
-140737488355329 -3 -140737488355332 -140737488355326 422212465065987 46912496118443 -0
-140737488355329 -2 -140737488355331 -140737488355327 281474976710658 70368744177664.5 -1
-140737488355329 -1 -140737488355330 -140737488355328 140737488355329 140737488355329 -0
-140737488355329 -0 -140737488355329 -140737488355329 +0 Infinity NaN
-140737488355329 +0 -140737488355329 -140737488355329 -0 -Infinity NaN
-140737488355329 1 -140737488355328 -140737488355330 -140737488355329 -140737488355329 -0
-140737488355329 2 -140737488355327 -140737488355331 -281474976710658 -70368744177664.5 -1
-140737488355329 3 -140737488355326 -140737488355332 -422212465065987 -46912496118443 -0
-140737488355328 -18446744073709552000 -18446884811197907000 18446603336221196000 2.596148429267414e+33 0.00000762939453125 -140737488355328
-140737488355328 -9223372036854776000 -9223512774343131000 9223231299366420000 1.298074214633707e+33 0.0000152587890625 -140737488355328
-140737488355328 -9223372036854776000 -9223512774343131000 9223231299366420000 1.298074214633707e+33 0.0000152587890625 -140737488355328
-140737488355328 -140737488355329 -281474976710657 1 1.9807040628566225e+28 0.9999999999999929 -140737488355328
-140737488355328 -140737488355328 -281474976710656 +0 1.9807040628566084e+28 1 -0
-140737488355328 140737488355327 -1 -281474976710655 -1.9807040628565944e+28 -1.000000000000007 -1
-140737488355328 140737488355328 +0 -281474976710656 -1.9807040628566084e+28 -1 -0
-140737488355328 9223372036854776000 9223231299366420000 -9223512774343131000 -1.298074214633707e+33 -0.0000152587890625 -140737488355328
-140737488355328 9223372036854776000 9223231299366420000 -9223512774343131000 -1.298074214633707e+33 -0.0000152587890625 -140737488355328
-140737488355328 18446744073709552000 18446603336221196000 -18446884811197907000 -2.596148429267414e+33 -0.00000762939453125 -140737488355328
-140737488355328 -4294967295 -140741783322623 -140733193388033 6.044629096665771e+23 32768.000007629395 -32768
-140737488355328 -2147483649 -140739635838977 -140735340871679 3.022314550443948e+23 65535.99996948242 -2147418113
-140737488355328 -2147483648 -140739635838976 -140735340871680 3.022314549036573e+23 65536 -0
-140737488355328 -2147483647 -140739635838975 -140735340871681 3.022314547629198e+23 65536.00003051758 -65536
-140737488355328 2147483647 -140735340871681 -140739635838975 -3.022314547629198e+23 -65536.00003051758 -65536
-140737488355328 2147483648 -140735340871680 -140739635838976 -3.022314549036573e+23 -65536 -0
-140737488355328 2147483649 -140735340871679 -140739635838977 -3.022314550443948e+23 -65535.99996948242 -2147418113
-140737488355328 4294967295 -140733193388033 -140741783322623 -6.044629096665771e+23 -32768.000007629395 -32768
-140737488355328 -3 -140737488355331 -140737488355325 422212465065984 46912496118442.664 -2
-140737488355328 -2 -140737488355330 -140737488355326 281474976710656 70368744177664 -0
-140737488355328 -1 -140737488355329 -140737488355327 140737488355328 140737488355328 -0
-140737488355328 -0 -140737488355328 -140737488355328 +0 Infinity NaN
-140737488355328 +0 -140737488355328 -140737488355328 -0 -Infinity NaN
-140737488355328 1 -140737488355327 -140737488355329 -140737488355328 -140737488355328 -0
-140737488355328 2 -140737488355326 -140737488355330 -281474976710656 -70368744177664 -0
-140737488355328 3 -140737488355325 -140737488355331 -422212465065984 -46912496118442.664 -2
140737488355327 -18446744073709552000 -18446603336221196000 18446884811197907000 -2.5961484292673954e+33 -0.000007629394531249946 140737488355327
140737488355327 -9223372036854776000 -9223231299366420000 9223512774343131000 -1.2980742146336977e+33 -0.000015258789062499892 140737488355327
140737488355327 -9223372036854776000 -9223231299366420000 9223512774343131000 -1.2980742146336977e+33 -0.000015258789062499892 140737488355327
140737488355327 -140737488355329 -2 281474976710656 -1.9807040628566084e+28 -0.9999999999999858 140737488355327
140737488355327 -140737488355328 -1 281474976710655 -1.9807040628565944e+28 -0.9999999999999929 140737488355327
140737488355327 140737488355327 281474976710654 +0 1.9807040628565803e+28 1 +0
140737488355327 140737488355328 281474976710655 -1 1.9807040628565944e+28 0.9999999999999929 140737488355327
140737488355327 9223372036854776000 9223512774343131000 -9223231299366420000 1.2980742146336977e+33 0.000015258789062499892 140737488355327
140737488355327 9223372036854776000 9223512774343131000 -9223231299366420000 1.2980742146336977e+33 0.000015258789062499892 140737488355327
140737488355327 18446744073709552000 18446884811197907000 -18446603336221196000 2.5961484292673954e+33 0.000007629394531249946 140737488355327
140737488355327 -4294967295 140733193388032 140741783322622 -6.044629096665728e+23 -32768.00000762916 32767
140737488355327 -2147483649 140735340871678 140739635838976 -3.0223145504439263e+23 -65535.999969481956 2147418112
140737488355327 -2147483648 140735340871679 140739635838975 -3.0223145490365515e+23 -65535.999999999534 2147483647
140737488355327 -2147483647 140735340871680 140739635838974 -3.0223145476291766e+23 -65536.00003051711 65535
140737488355327 2147483647 140739635838974 140735340871680 3.0223145476291766e+23 65536.00003051711 65535
140737488355327 2147483648 140739635838975 140735340871679 3.0223145490365515e+23 65535.999999999534 2147483647
140737488355327 2147483649 140739635838976 140735340871678 3.0223145504439263e+23 65535.999969481956 2147418112
140737488355327 4294967295 140741783322622 140733193388032 6.044629096665728e+23 32768.00000762916 32767
140737488355327 -3 140737488355324 140737488355330 -422212465065981 -46912496118442.336 1
140737488355327 -2 140737488355325 140737488355329 -281474976710654 -70368744177663.5 1
140737488355327 -1 140737488355326 140737488355328 -140737488355327 -140737488355327 +0
140737488355327 -0 140737488355327 140737488355327 -0 -Infinity NaN
140737488355327 +0 140737488355327 140737488355327 +0 Infinity NaN
140737488355327 1 140737488355328 140737488355326 140737488355327 140737488355327 +0
140737488355327 2 140737488355329 140737488355325 281474976710654 70368744177663.5 1
140737488355327 3 140737488355330 140737488355324 422212465065981 46912496118442.336 1
140737488355328 -18446744073709552000 -18446603336221196000 18446884811197907000 -2.596148429267414e+33 -0.00000762939453125 140737488355328
140737488355328 -9223372036854776000 -9223231299366420000 9223512774343131000 -1.298074214633707e+33 -0.0000152587890625 140737488355328
140737488355328 -9223372036854776000 -9223231299366420000 9223512774343131000 -1.298074214633707e+33 -0.0000152587890625 140737488355328
140737488355328 -140737488355329 -1 281474976710657 -1.9807040628566225e+28 -0.9999999999999929 140737488355328
140737488355328 -140737488355328 +0 281474976710656 -1.9807040628566084e+28 -1 +0
140737488355328 140737488355327 281474976710655 1 1.9807040628565944e+28 1.000000000000007 1
140737488355328 140737488355328 281474976710656 +0 1.9807040628566084e+28 1 +0
140737488355328 9223372036854776000 9223512774343131000 -9223231299366420000 1.298074214633707e+33 0.0000152587890625 140737488355328
140737488355328 9223372036854776000 9223512774343131000 -9223231299366420000 1.298074214633707e+33 0.0000152587890625 140737488355328
140737488355328 18446744073709552000 18446884811197907000 -18446603336221196000 2.596148429267414e+33 0.00000762939453125 140737488355328
140737488355328 -4294967295 140733193388033 140741783322623 -6.044629096665771e+23 -32768.000007629395 32768
140737488355328 -2147483649 140735340871679 140739635838977 -3.022314550443948e+23 -65535.99996948242 2147418113
140737488355328 -2147483648 140735340871680 140739635838976 -3.022314549036573e+23 -65536 +0
140737488355328 -2147483647 140735340871681 140739635838975 -3.022314547629198e+23 -65536.00003051758 65536
140737488355328 2147483647 140739635838975 140735340871681 3.022314547629198e+23 65536.00003051758 65536
140737488355328 2147483648 140739635838976 140735340871680 3.022314549036573e+23 65536 +0
140737488355328 2147483649 140739635838977 140735340871679 3.022314550443948e+23 65535.99996948242 2147418113
140737488355328 4294967295 140741783322623 140733193388033 6.044629096665771e+23 32768.000007629395 32768
140737488355328 -3 140737488355325 140737488355331 -422212465065984 -46912496118442.664 2
140737488355328 -2 140737488355326 140737488355330 -281474976710656 -70368744177664 +0
140737488355328 -1 140737488355327 140737488355329 -140737488355328 -140737488355328 +0
140737488355328 -0 140737488355328 140737488355328 -0 -Infinity NaN
140737488355328 +0 140737488355328 140737488355328 +0 Infinity NaN
140737488355328 1 140737488355329 140737488355327 140737488355328 140737488355328 +0
140737488355328 2 140737488355330 140737488355326 281474976710656 70368744177664 +0
140737488355328 3 140737488355331 140737488355325 422212465065984 46912496118442.664 2
9223372036854776000 -18446744073709552000 -9223372036854776000 27670116110564327000 -1.7014118346046923e+38 -0.5 9223372036854776000
9223372036854776000 -9223372036854776000 +0 18446744073709552000 -8.507059173023462e+37 -1 +0
9223372036854776000 -9223372036854776000 +0 18446744073709552000 -8.507059173023462e+37 -1 +0
9223372036854776000 -140737488355329 9223231299366420000 9223512774343131000 -1.298074214633716e+33 -65535.999999999534 140737488289793
9223372036854776000 -140737488355328 9223231299366420000 9223512774343131000 -1.298074214633707e+33 -65536 +0
9223372036854776000 140737488355327 9223512774343131000 9223231299366420000 1.2980742146336977e+33 65536.00000000047 65536
9223372036854776000 140737488355328 9223512774343131000 9223231299366420000 1.298074214633707e+33 65536 +0
9223372036854776000 9223372036854776000 18446744073709552000 +0 8.507059173023462e+37 1 +0
9223372036854776000 9223372036854776000 18446744073709552000 +0 8.507059173023462e+37 1 +0
9223372036854776000 18446744073709552000 27670116110564327000 -9223372036854776000 1.7014118346046923e+38 0.5 9223372036854776000
9223372036854776000 -4294967295 9223372032559809000 9223372041149743000 -3.9614081247908797e+28 -2147483648.5 2147483648
9223372036854776000 -2147483649 9223372034707292000 9223372039002260000 -1.9807040637789456e+28 -4294967294 2
9223372036854776000 -2147483648 9223372034707292000 9223372039002260000 -1.9807040628566084e+28 -4294967296 +0
9223372036854776000 -2147483647 9223372034707292000 9223372039002260000 -1.9807040619342712e+28 -4294967298 2
9223372036854776000 2147483647 9223372039002260000 9223372034707292000 1.9807040619342712e+28 4294967298 2
9223372036854776000 2147483648 9223372039002260000 9223372034707292000 1.9807040628566084e+28 4294967296 +0
9223372036854776000 2147483649 9223372039002260000 9223372034707292000 1.9807040637789456e+28 4294967294 2
9223372036854776000 4294967295 9223372041149743000 9223372032559809000 3.9614081247908797e+28 2147483648.5 2147483648
9223372036854776000 -3 9223372036854776000 9223372036854776000 -27670116110564327000 -3074457345618258400 2
9223372036854776000 -2 9223372036854776000 9223372036854776000 -18446744073709552000 -4611686018427388000 +0
9223372036854776000 -1 9223372036854776000 9223372036854776000 -9223372036854776000 -9223372036854776000 +0
9223372036854776000 -0 9223372036854776000 9223372036854776000 -0 -Infinity NaN
9223372036854776000 +0 9223372036854776000 9223372036854776000 +0 Infinity NaN
9223372036854776000 1 9223372036854776000 9223372036854776000 9223372036854776000 9223372036854776000 +0
9223372036854776000 2 9223372036854776000 9223372036854776000 18446744073709552000 4611686018427388000 +0
9223372036854776000 3 9223372036854776000 9223372036854776000 27670116110564327000 3074457345618258400 2
9223372036854776000 -18446744073709552000 -9223372036854776000 27670116110564327000 -1.7014118346046923e+38 -0.5 9223372036854776000
9223372036854776000 -9223372036854776000 +0 18446744073709552000 -8.507059173023462e+37 -1 +0
9223372036854776000 -9223372036854776000 +0 18446744073709552000 -8.507059173023462e+37 -1 +0
9223372036854776000 -140737488355329 9223231299366420000 9223512774343131000 -1.298074214633716e+33 -65535.999999999534 140737488289793
9223372036854776000 -140737488355328 9223231299366420000 9223512774343131000 -1.298074214633707e+33 -65536 +0
9223372036854776000 140737488355327 9223512774343131000 9223231299366420000 1.2980742146336977e+33 65536.00000000047 65536
9223372036854776000 140737488355328 9223512774343131000 9223231299366420000 1.298074214633707e+33 65536 +0
9223372036854776000 9223372036854776000 18446744073709552000 +0 8.507059173023462e+37 1 +0
9223372036854776000 9223372036854776000 18446744073709552000 +0 8.507059173023462e+37 1 +0
9223372036854776000 18446744073709552000 27670116110564327000 -9223372036854776000 1.7014118346046923e+38 0.5 9223372036854776000
9223372036854776000 -4294967295 9223372032559809000 9223372041149743000 -3.9614081247908797e+28 -2147483648.5 2147483648
9223372036854776000 -2147483649 9223372034707292000 9223372039002260000 -1.9807040637789456e+28 -4294967294 2
9223372036854776000 -2147483648 9223372034707292000 9223372039002260000 -1.9807040628566084e+28 -4294967296 +0
9223372036854776000 -2147483647 9223372034707292000 9223372039002260000 -1.9807040619342712e+28 -4294967298 2
9223372036854776000 2147483647 9223372039002260000 9223372034707292000 1.9807040619342712e+28 4294967298 2
9223372036854776000 2147483648 9223372039002260000 9223372034707292000 1.9807040628566084e+28 4294967296 +0
9223372036854776000 2147483649 9223372039002260000 9223372034707292000 1.9807040637789456e+28 4294967294 2
9223372036854776000 4294967295 9223372041149743000 9223372032559809000 3.9614081247908797e+28 2147483648.5 2147483648
9223372036854776000 -3 9223372036854776000 9223372036854776000 -27670116110564327000 -3074457345618258400 2
9223372036854776000 -2 9223372036854776000 9223372036854776000 -18446744073709552000 -4611686018427388000 +0
9223372036854776000 -1 9223372036854776000 9223372036854776000 -9223372036854776000 -9223372036854776000 +0
9223372036854776000 -0 9223372036854776000 9223372036854776000 -0 -Infinity NaN
9223372036854776000 +0 9223372036854776000 9223372036854776000 +0 Infinity NaN
9223372036854776000 1 9223372036854776000 9223372036854776000 9223372036854776000 9223372036854776000 +0
9223372036854776000 2 9223372036854776000 9223372036854776000 18446744073709552000 4611686018427388000 +0
9223372036854776000 3 9223372036854776000 9223372036854776000 27670116110564327000 3074457345618258400 2
18446744073709552000 -18446744073709552000 +0 36893488147419103000 -3.402823669209385e+38 -1 +0
18446744073709552000 -9223372036854776000 9223372036854776000 27670116110564327000 -1.7014118346046923e+38 -2 +0
18446744073709552000 -9223372036854776000 9223372036854776000 27670116110564327000 -1.7014118346046923e+38 -2 +0
18446744073709552000 -140737488355329 18446603336221196000 18446884811197907000 -2.596148429267432e+33 -131071.99999999907 140737488224257
18446744073709552000 -140737488355328 18446603336221196000 18446884811197907000 -2.596148429267414e+33 -131072 +0
18446744073709552000 140737488355327 18446884811197907000 18446603336221196000 2.5961484292673954e+33 131072.00000000093 131072
18446744073709552000 140737488355328 18446884811197907000 18446603336221196000 2.596148429267414e+33 131072 +0
18446744073709552000 9223372036854776000 27670116110564327000 9223372036854776000 1.7014118346046923e+38 2 +0
18446744073709552000 9223372036854776000 27670116110564327000 9223372036854776000 1.7014118346046923e+38 2 +0
18446744073709552000 18446744073709552000 36893488147419103000 +0 3.402823669209385e+38 1 +0
18446744073709552000 -4294967295 18446744069414584000 18446744078004520000 -7.922816249581759e+28 -4294967297 1
18446744073709552000 -2147483649 18446744071562068000 18446744075857035000 -3.961408127557891e+28 -8589934588 4
18446744073709552000 -2147483648 18446744071562068000 18446744075857035000 -3.961408125713217e+28 -8589934592 +0
18446744073709552000 -2147483647 18446744071562068000 18446744075857035000 -3.9614081238685425e+28 -8589934596 4
18446744073709552000 2147483647 18446744075857035000 18446744071562068000 3.9614081238685425e+28 8589934596 4
18446744073709552000 2147483648 18446744075857035000 18446744071562068000 3.961408125713217e+28 8589934592 +0
18446744073709552000 2147483649 18446744075857035000 18446744071562068000 3.961408127557891e+28 8589934588 4
18446744073709552000 4294967295 18446744078004520000 18446744069414584000 7.922816249581759e+28 4294967297 1
18446744073709552000 -3 18446744073709552000 18446744073709552000 -55340232221128655000 -6148914691236517000 1
18446744073709552000 -2 18446744073709552000 18446744073709552000 -36893488147419103000 -9223372036854776000 +0
18446744073709552000 -1 18446744073709552000 18446744073709552000 -18446744073709552000 -18446744073709552000 +0
18446744073709552000 -0 18446744073709552000 18446744073709552000 -0 -Infinity NaN
18446744073709552000 +0 18446744073709552000 18446744073709552000 +0 Infinity NaN
18446744073709552000 1 18446744073709552000 18446744073709552000 18446744073709552000 18446744073709552000 +0
18446744073709552000 2 18446744073709552000 18446744073709552000 36893488147419103000 9223372036854776000 +0
18446744073709552000 3 18446744073709552000 18446744073709552000 55340232221128655000 6148914691236517000 1
-4294967295 -18446744073709552000 -18446744078004520000 18446744069414584000 7.922816249581759e+28 2.3283064359965952e-10 -4294967295
-4294967295 -9223372036854776000 -9223372041149743000 9223372032559809000 3.9614081247908797e+28 4.6566128719931904e-10 -4294967295
-4294967295 -9223372036854776000 -9223372041149743000 9223372032559809000 3.9614081247908797e+28 4.6566128719931904e-10 -4294967295
-4294967295 -140737488355329 -140741783322624 140733193388034 6.044629096665814e+23 0.000030517578117894356 -4294967295
-4294967295 -140737488355328 -140741783322623 140733193388033 6.044629096665771e+23 0.000030517578117894573 -4294967295
-4294967295 140737488355327 140733193388032 -140741783322622 -6.044629096665728e+23 -0.00003051757811789479 -4294967295
-4294967295 140737488355328 140733193388033 -140741783322623 -6.044629096665771e+23 -0.000030517578117894573 -4294967295
-4294967295 9223372036854776000 9223372032559809000 -9223372041149743000 -3.9614081247908797e+28 -4.6566128719931904e-10 -4294967295
-4294967295 9223372036854776000 9223372032559809000 -9223372041149743000 -3.9614081247908797e+28 -4.6566128719931904e-10 -4294967295
-4294967295 18446744073709552000 18446744069414584000 -18446744078004520000 -7.922816249581759e+28 -2.3283064359965952e-10 -4294967295
-4294967295 -4294967295 -8589934590 +0 18446744065119617000 1 -0
-4294967295 -2147483649 -6442450944 -2147483646 9223372039002260000 1.9999999986030161 -2147483646
-4294967295 -2147483648 -6442450943 -2147483647 9223372034707292000 1.9999999995343387 -2147483647
-4294967295 -2147483647 -6442450942 -2147483648 9223372030412325000 2.0000000004656613 -1
-4294967295 2147483647 -2147483648 -6442450942 -9223372030412325000 -2.0000000004656613 -1
-4294967295 2147483648 -2147483647 -6442450943 -9223372034707292000 -1.9999999995343387 -2147483647
-4294967295 2147483649 -2147483646 -6442450944 -9223372039002260000 -1.9999999986030161 -2147483646
-4294967295 4294967295 +0 -8589934590 -18446744065119617000 -1 -0
-4294967295 -3 -4294967298 -4294967292 12884901885 1431655765 -0
-4294967295 -2 -4294967297 -4294967293 8589934590 2147483647.5 -1
-4294967295 -1 -4294967296 -4294967294 4294967295 4294967295 -0
-4294967295 -0 -4294967295 -4294967295 +0 Infinity NaN
-4294967295 +0 -4294967295 -4294967295 -0 -Infinity NaN
-4294967295 1 -4294967294 -4294967296 -4294967295 -4294967295 -0
-4294967295 2 -4294967293 -4294967297 -8589934590 -2147483647.5 -1
-4294967295 3 -4294967292 -4294967298 -12884901885 -1431655765 -0
-2147483649 -18446744073709552000 -18446744075857035000 18446744071562068000 3.961408127557891e+28 1.1641532188114492e-10 -2147483649
-2147483649 -9223372036854776000 -9223372039002260000 9223372034707292000 1.9807040637789456e+28 2.3283064376228985e-10 -2147483649
-2147483649 -9223372036854776000 -9223372039002260000 9223372034707292000 1.9807040637789456e+28 2.3283064376228985e-10 -2147483649
-2147483649 -140737488355329 -140739635838978 140735340871680 3.022314550443969e+23 0.00001525878906960532 -2147483649
-2147483649 -140737488355328 -140739635838977 140735340871679 3.022314550443948e+23 0.000015258789069605427 -2147483649
-2147483649 140737488355327 140735340871678 -140739635838976 -3.0223145504439263e+23 -0.000015258789069605536 -2147483649
-2147483649 140737488355328 140735340871679 -140739635838977 -3.022314550443948e+23 -0.000015258789069605427 -2147483649
-2147483649 9223372036854776000 9223372034707292000 -9223372039002260000 -1.9807040637789456e+28 -2.3283064376228985e-10 -2147483649
-2147483649 9223372036854776000 9223372034707292000 -9223372039002260000 -1.9807040637789456e+28 -2.3283064376228985e-10 -2147483649
-2147483649 18446744073709552000 18446744071562068000 -18446744075857035000 -3.961408127557891e+28 -1.1641532188114492e-10 -2147483649
-2147483649 -4294967295 -6442450944 2147483646 9223372039002260000 0.500000000349246 -2147483649
-2147483649 -2147483649 -4294967298 +0 4611686022722355000 1 -0
-2147483649 -2147483648 -4294967297 -1 4611686020574872000 1.0000000004656613 -1
-2147483649 -2147483647 -4294967296 -2 4611686018427388000 1.0000000009313226 -2
-2147483649 2147483647 -2 -4294967296 -4611686018427388000 -1.0000000009313226 -2
-2147483649 2147483648 -1 -4294967297 -4611686020574872000 -1.0000000004656613 -1
-2147483649 2147483649 +0 -4294967298 -4611686022722355000 -1 -0
-2147483649 4294967295 2147483646 -6442450944 -9223372039002260000 -0.500000000349246 -2147483649
-2147483649 -3 -2147483652 -2147483646 6442450947 715827883 -0
-2147483649 -2 -2147483651 -2147483647 4294967298 1073741824.5 -1
-2147483649 -1 -2147483650 -2147483648 2147483649 2147483649 -0
-2147483649 -0 -2147483649 -2147483649 +0 Infinity NaN
-2147483649 +0 -2147483649 -2147483649 -0 -Infinity NaN
-2147483649 1 -2147483648 -2147483650 -2147483649 -2147483649 -0
-2147483649 2 -2147483647 -2147483651 -4294967298 -1073741824.5 -1
-2147483649 3 -2147483646 -2147483652 -6442450947 -715827883 -0
-2147483648 -18446744073709552000 -18446744075857035000 18446744071562068000 3.961408125713217e+28 1.1641532182693481e-10 -2147483648
-2147483648 -9223372036854776000 -9223372039002260000 9223372034707292000 1.9807040628566084e+28 2.3283064365386963e-10 -2147483648
-2147483648 -9223372036854776000 -9223372039002260000 9223372034707292000 1.9807040628566084e+28 2.3283064365386963e-10 -2147483648
-2147483648 -140737488355329 -140739635838977 140735340871681 3.0223145490365944e+23 0.000015258789062499892 -2147483648
-2147483648 -140737488355328 -140739635838976 140735340871680 3.022314549036573e+23 0.0000152587890625 -2147483648
-2147483648 140737488355327 140735340871679 -140739635838975 -3.0223145490365515e+23 -0.00001525878906250011 -2147483648
-2147483648 140737488355328 140735340871680 -140739635838976 -3.022314549036573e+23 -0.0000152587890625 -2147483648
-2147483648 9223372036854776000 9223372034707292000 -9223372039002260000 -1.9807040628566084e+28 -2.3283064365386963e-10 -2147483648
-2147483648 9223372036854776000 9223372034707292000 -9223372039002260000 -1.9807040628566084e+28 -2.3283064365386963e-10 -2147483648
-2147483648 18446744073709552000 18446744071562068000 -18446744075857035000 -3.961408125713217e+28 -1.1641532182693481e-10 -2147483648
-2147483648 -4294967295 -6442450943 2147483647 9223372034707292000 0.5000000001164153 -2147483648
-2147483648 -2147483649 -4294967297 1 4611686020574872000 0.9999999995343387 -2147483648
-2147483648 -2147483648 -4294967296 +0 4611686018427388000 1 -0
-2147483648 -2147483647 -4294967295 -1 4611686016279904000 1.0000000004656613 -1
-2147483648 2147483647 -1 -4294967295 -4611686016279904000 -1.0000000004656613 -1
-2147483648 2147483648 +0 -4294967296 -4611686018427388000 -1 -0
-2147483648 2147483649 1 -4294967297 -4611686020574872000 -0.9999999995343387 -2147483648
-2147483648 4294967295 2147483647 -6442450943 -9223372034707292000 -0.5000000001164153 -2147483648
-2147483648 -3 -2147483651 -2147483645 6442450944 715827882.6666666 -2
-2147483648 -2 -2147483650 -2147483646 4294967296 1073741824 -0
-2147483648 -1 -2147483649 -2147483647 2147483648 2147483648 -0
-2147483648 -0 -2147483648 -2147483648 +0 Infinity NaN
-2147483648 +0 -2147483648 -2147483648 -0 -Infinity NaN
-2147483648 1 -2147483647 -2147483649 -2147483648 -2147483648 -0
-2147483648 2 -2147483646 -2147483650 -4294967296 -1073741824 -0
-2147483648 3 -2147483645 -2147483651 -6442450944 -715827882.6666666 -2
-2147483647 -18446744073709552000 -18446744075857035000 18446744071562068000 3.9614081238685425e+28 1.164153217727247e-10 -2147483647
-2147483647 -9223372036854776000 -9223372039002260000 9223372034707292000 1.9807040619342712e+28 2.328306435454494e-10 -2147483647
-2147483647 -9223372036854776000 -9223372039002260000 9223372034707292000 1.9807040619342712e+28 2.328306435454494e-10 -2147483647
-2147483647 -140737488355329 -140739635838976 140735340871682 3.0223145476292195e+23 0.000015258789055394464 -2147483647
-2147483647 -140737488355328 -140739635838975 140735340871681 3.022314547629198e+23 0.000015258789055394573 -2147483647
-2147483647 140737488355327 140735340871680 -140739635838974 -3.0223145476291766e+23 -0.000015258789055394681 -2147483647
-2147483647 140737488355328 140735340871681 -140739635838975 -3.022314547629198e+23 -0.000015258789055394573 -2147483647
-2147483647 9223372036854776000 9223372034707292000 -9223372039002260000 -1.9807040619342712e+28 -2.328306435454494e-10 -2147483647
-2147483647 9223372036854776000 9223372034707292000 -9223372039002260000 -1.9807040619342712e+28 -2.328306435454494e-10 -2147483647
-2147483647 18446744073709552000 18446744071562068000 -18446744075857035000 -3.9614081238685425e+28 -1.164153217727247e-10 -2147483647
-2147483647 -4294967295 -6442450942 2147483648 9223372030412325000 0.4999999998835847 -2147483647
-2147483647 -2147483649 -4294967296 2 4611686018427388000 0.9999999990686774 -2147483647
-2147483647 -2147483648 -4294967295 1 4611686016279904000 0.9999999995343387 -2147483647
-2147483647 -2147483647 -4294967294 +0 4611686014132420600 1 -0
-2147483647 2147483647 +0 -4294967294 -4611686014132420600 -1 -0
-2147483647 2147483648 1 -4294967295 -4611686016279904000 -0.9999999995343387 -2147483647
-2147483647 2147483649 2 -4294967296 -4611686018427388000 -0.9999999990686774 -2147483647
-2147483647 4294967295 2147483648 -6442450942 -9223372030412325000 -0.4999999998835847 -2147483647
-2147483647 -3 -2147483650 -2147483644 6442450941 715827882.3333334 -1
-2147483647 -2 -2147483649 -2147483645 4294967294 1073741823.5 -1
-2147483647 -1 -2147483648 -2147483646 2147483647 2147483647 -0
-2147483647 -0 -2147483647 -2147483647 +0 Infinity NaN
-2147483647 +0 -2147483647 -2147483647 -0 -Infinity NaN
-2147483647 1 -2147483646 -2147483648 -2147483647 -2147483647 -0
-2147483647 2 -2147483645 -2147483649 -4294967294 -1073741823.5 -1
-2147483647 3 -2147483644 -2147483650 -6442450941 -715827882.3333334 -1
2147483647 -18446744073709552000 -18446744071562068000 18446744075857035000 -3.9614081238685425e+28 -1.164153217727247e-10 2147483647
2147483647 -9223372036854776000 -9223372034707292000 9223372039002260000 -1.9807040619342712e+28 -2.328306435454494e-10 2147483647
2147483647 -9223372036854776000 -9223372034707292000 9223372039002260000 -1.9807040619342712e+28 -2.328306435454494e-10 2147483647
2147483647 -140737488355329 -140735340871682 140739635838976 -3.0223145476292195e+23 -0.000015258789055394464 2147483647
2147483647 -140737488355328 -140735340871681 140739635838975 -3.022314547629198e+23 -0.000015258789055394573 2147483647
2147483647 140737488355327 140739635838974 -140735340871680 3.0223145476291766e+23 0.000015258789055394681 2147483647
2147483647 140737488355328 140739635838975 -140735340871681 3.022314547629198e+23 0.000015258789055394573 2147483647
2147483647 9223372036854776000 9223372039002260000 -9223372034707292000 1.9807040619342712e+28 2.328306435454494e-10 2147483647
2147483647 9223372036854776000 9223372039002260000 -9223372034707292000 1.9807040619342712e+28 2.328306435454494e-10 2147483647
2147483647 18446744073709552000 18446744075857035000 -18446744071562068000 3.9614081238685425e+28 1.164153217727247e-10 2147483647
2147483647 -4294967295 -2147483648 6442450942 -9223372030412325000 -0.4999999998835847 2147483647
2147483647 -2147483649 -2 4294967296 -4611686018427388000 -0.9999999990686774 2147483647
2147483647 -2147483648 -1 4294967295 -4611686016279904000 -0.9999999995343387 2147483647
2147483647 -2147483647 +0 4294967294 -4611686014132420600 -1 +0
2147483647 2147483647 4294967294 +0 4611686014132420600 1 +0
2147483647 2147483648 4294967295 -1 4611686016279904000 0.9999999995343387 2147483647
2147483647 2147483649 4294967296 -2 4611686018427388000 0.9999999990686774 2147483647
2147483647 4294967295 6442450942 -2147483648 9223372030412325000 0.4999999998835847 2147483647
2147483647 -3 2147483644 2147483650 -6442450941 -715827882.3333334 1
2147483647 -2 2147483645 2147483649 -4294967294 -1073741823.5 1
2147483647 -1 2147483646 2147483648 -2147483647 -2147483647 +0
2147483647 -0 2147483647 2147483647 -0 -Infinity NaN
2147483647 +0 2147483647 2147483647 +0 Infinity NaN
2147483647 1 2147483648 2147483646 2147483647 2147483647 +0
2147483647 2 2147483649 2147483645 4294967294 1073741823.5 1
2147483647 3 2147483650 2147483644 6442450941 715827882.3333334 1
2147483648 -18446744073709552000 -18446744071562068000 18446744075857035000 -3.961408125713217e+28 -1.1641532182693481e-10 2147483648
2147483648 -9223372036854776000 -9223372034707292000 9223372039002260000 -1.9807040628566084e+28 -2.3283064365386963e-10 2147483648
2147483648 -9223372036854776000 -9223372034707292000 9223372039002260000 -1.9807040628566084e+28 -2.3283064365386963e-10 2147483648
2147483648 -140737488355329 -140735340871681 140739635838977 -3.0223145490365944e+23 -0.000015258789062499892 2147483648
2147483648 -140737488355328 -140735340871680 140739635838976 -3.022314549036573e+23 -0.0000152587890625 2147483648
2147483648 140737488355327 140739635838975 -140735340871679 3.0223145490365515e+23 0.00001525878906250011 2147483648
2147483648 140737488355328 140739635838976 -140735340871680 3.022314549036573e+23 0.0000152587890625 2147483648
2147483648 9223372036854776000 9223372039002260000 -9223372034707292000 1.9807040628566084e+28 2.3283064365386963e-10 2147483648
2147483648 9223372036854776000 9223372039002260000 -9223372034707292000 1.9807040628566084e+28 2.3283064365386963e-10 2147483648
2147483648 18446744073709552000 18446744075857035000 -18446744071562068000 3.961408125713217e+28 1.1641532182693481e-10 2147483648
2147483648 -4294967295 -2147483647 6442450943 -9223372034707292000 -0.5000000001164153 2147483648
2147483648 -2147483649 -1 4294967297 -4611686020574872000 -0.9999999995343387 2147483648
2147483648 -2147483648 +0 4294967296 -4611686018427388000 -1 +0
2147483648 -2147483647 1 4294967295 -4611686016279904000 -1.0000000004656613 1
2147483648 2147483647 4294967295 1 4611686016279904000 1.0000000004656613 1
2147483648 2147483648 4294967296 +0 4611686018427388000 1 +0
2147483648 2147483649 4294967297 -1 4611686020574872000 0.9999999995343387 2147483648
2147483648 4294967295 6442450943 -2147483647 9223372034707292000 0.5000000001164153 2147483648
2147483648 -3 2147483645 2147483651 -6442450944 -715827882.6666666 2
2147483648 -2 2147483646 2147483650 -4294967296 -1073741824 +0
2147483648 -1 2147483647 2147483649 -2147483648 -2147483648 +0
2147483648 -0 2147483648 2147483648 -0 -Infinity NaN
2147483648 +0 2147483648 2147483648 +0 Infinity NaN
2147483648 1 2147483649 2147483647 2147483648 2147483648 +0
2147483648 2 2147483650 2147483646 4294967296 1073741824 +0
2147483648 3 2147483651 2147483645 6442450944 715827882.6666666 2
2147483649 -18446744073709552000 -18446744071562068000 18446744075857035000 -3.961408127557891e+28 -1.1641532188114492e-10 2147483649
2147483649 -9223372036854776000 -9223372034707292000 9223372039002260000 -1.9807040637789456e+28 -2.3283064376228985e-10 2147483649
2147483649 -9223372036854776000 -9223372034707292000 9223372039002260000 -1.9807040637789456e+28 -2.3283064376228985e-10 2147483649
2147483649 -140737488355329 -140735340871680 140739635838978 -3.022314550443969e+23 -0.00001525878906960532 2147483649
2147483649 -140737488355328 -140735340871679 140739635838977 -3.022314550443948e+23 -0.000015258789069605427 2147483649
2147483649 140737488355327 140739635838976 -140735340871678 3.0223145504439263e+23 0.000015258789069605536 2147483649
2147483649 140737488355328 140739635838977 -140735340871679 3.022314550443948e+23 0.000015258789069605427 2147483649
2147483649 9223372036854776000 9223372039002260000 -9223372034707292000 1.9807040637789456e+28 2.3283064376228985e-10 2147483649
2147483649 9223372036854776000 9223372039002260000 -9223372034707292000 1.9807040637789456e+28 2.3283064376228985e-10 2147483649
2147483649 18446744073709552000 18446744075857035000 -18446744071562068000 3.961408127557891e+28 1.1641532188114492e-10 2147483649
2147483649 -4294967295 -2147483646 6442450944 -9223372039002260000 -0.500000000349246 2147483649
2147483649 -2147483649 +0 4294967298 -4611686022722355000 -1 +0
2147483649 -2147483648 1 4294967297 -4611686020574872000 -1.0000000004656613 1
2147483649 -2147483647 2 4294967296 -4611686018427388000 -1.0000000009313226 2
2147483649 2147483647 4294967296 2 4611686018427388000 1.0000000009313226 2
2147483649 2147483648 4294967297 1 4611686020574872000 1.0000000004656613 1
2147483649 2147483649 4294967298 +0 4611686022722355000 1 +0
2147483649 4294967295 6442450944 -2147483646 9223372039002260000 0.500000000349246 2147483649
2147483649 -3 2147483646 2147483652 -6442450947 -715827883 +0
2147483649 -2 2147483647 2147483651 -4294967298 -1073741824.5 1
2147483649 -1 2147483648 2147483650 -2147483649 -2147483649 +0
2147483649 -0 2147483649 2147483649 -0 -Infinity NaN
2147483649 +0 2147483649 2147483649 +0 Infinity NaN
2147483649 1 2147483650 2147483648 2147483649 2147483649 +0
2147483649 2 2147483651 2147483647 4294967298 1073741824.5 1
2147483649 3 2147483652 2147483646 6442450947 715827883 +0
4294967295 -18446744073709552000 -18446744069414584000 18446744078004520000 -7.922816249581759e+28 -2.3283064359965952e-10 4294967295
4294967295 -9223372036854776000 -9223372032559809000 9223372041149743000 -3.9614081247908797e+28 -4.6566128719931904e-10 4294967295
4294967295 -9223372036854776000 -9223372032559809000 9223372041149743000 -3.9614081247908797e+28 -4.6566128719931904e-10 4294967295
4294967295 -140737488355329 -140733193388034 140741783322624 -6.044629096665814e+23 -0.000030517578117894356 4294967295
4294967295 -140737488355328 -140733193388033 140741783322623 -6.044629096665771e+23 -0.000030517578117894573 4294967295
4294967295 140737488355327 140741783322622 -140733193388032 6.044629096665728e+23 0.00003051757811789479 4294967295
4294967295 140737488355328 140741783322623 -140733193388033 6.044629096665771e+23 0.000030517578117894573 4294967295
4294967295 9223372036854776000 9223372041149743000 -9223372032559809000 3.9614081247908797e+28 4.6566128719931904e-10 4294967295
4294967295 9223372036854776000 9223372041149743000 -9223372032559809000 3.9614081247908797e+28 4.6566128719931904e-10 4294967295
4294967295 18446744073709552000 18446744078004520000 -18446744069414584000 7.922816249581759e+28 2.3283064359965952e-10 4294967295
4294967295 -4294967295 +0 8589934590 -18446744065119617000 -1 +0
4294967295 -2147483649 2147483646 6442450944 -9223372039002260000 -1.9999999986030161 2147483646
4294967295 -2147483648 2147483647 6442450943 -9223372034707292000 -1.9999999995343387 2147483647
4294967295 -2147483647 2147483648 6442450942 -9223372030412325000 -2.0000000004656613 1
4294967295 2147483647 6442450942 2147483648 9223372030412325000 2.0000000004656613 1
4294967295 2147483648 6442450943 2147483647 9223372034707292000 1.9999999995343387 2147483647
4294967295 2147483649 6442450944 2147483646 9223372039002260000 1.9999999986030161 2147483646
4294967295 4294967295 8589934590 +0 18446744065119617000 1 +0
4294967295 -3 4294967292 4294967298 -12884901885 -1431655765 +0
4294967295 -2 4294967293 4294967297 -8589934590 -2147483647.5 1
4294967295 -1 4294967294 4294967296 -4294967295 -4294967295 +0
4294967295 -0 4294967295 4294967295 -0 -Infinity NaN
4294967295 +0 4294967295 4294967295 +0 Infinity NaN
4294967295 1 4294967296 4294967294 4294967295 4294967295 +0
4294967295 2 4294967297 4294967293 8589934590 2147483647.5 1
4294967295 3 4294967298 4294967292 12884901885 1431655765 +0
-3 -18446744073709552000 -18446744073709552000 18446744073709552000 55340232221128655000 1.6263032587282567e-19 -3
-3 -9223372036854776000 -9223372036854776000 9223372036854776000 27670116110564327000 3.2526065174565133e-19 -3
-3 -9223372036854776000 -9223372036854776000 9223372036854776000 27670116110564327000 3.2526065174565133e-19 -3
-3 -140737488355329 -140737488355332 140737488355326 422212465065987 2.1316282072802854e-14 -3
-3 -140737488355328 -140737488355331 140737488355325 422212465065984 2.1316282072803006e-14 -3
-3 140737488355327 140737488355324 -140737488355330 -422212465065981 -2.1316282072803157e-14 -3
-3 140737488355328 140737488355325 -140737488355331 -422212465065984 -2.1316282072803006e-14 -3
-3 9223372036854776000 9223372036854776000 -9223372036854776000 -27670116110564327000 -3.2526065174565133e-19 -3
-3 9223372036854776000 9223372036854776000 -9223372036854776000 -27670116110564327000 -3.2526065174565133e-19 -3
-3 18446744073709552000 18446744073709552000 -18446744073709552000 -55340232221128655000 -1.6263032587282567e-19 -3
-3 -4294967295 -4294967298 4294967292 12884901885 6.984919311242392e-10 -3
-3 -2147483649 -2147483652 2147483646 6442450947 1.3969838612726965e-9 -3
-3 -2147483648 -2147483651 2147483645 6442450944 1.3969838619232178e-9 -3
-3 -2147483647 -2147483650 2147483644 6442450941 1.396983862573739e-9 -3
-3 2147483647 2147483644 -2147483650 -6442450941 -1.396983862573739e-9 -3
-3 2147483648 2147483645 -2147483651 -6442450944 -1.3969838619232178e-9 -3
-3 2147483649 2147483646 -2147483652 -6442450947 -1.3969838612726965e-9 -3
-3 4294967295 4294967292 -4294967298 -12884901885 -6.984919311242392e-10 -3
-3 -3 -6 +0 9 1 -0
-3 -2 -5 -1 6 1.5 -1
-3 -1 -4 -2 3 3 -0
-3 -0 -3 -3 +0 Infinity NaN
-3 +0 -3 -3 -0 -Infinity NaN
-3 1 -2 -4 -3 -3 -0
-3 2 -1 -5 -6 -1.5 -1
-3 3 +0 -6 -9 -1 -0
-2 -18446744073709552000 -18446744073709552000 18446744073709552000 36893488147419103000 1.0842021724855044e-19 -2
-2 -9223372036854776000 -9223372036854776000 9223372036854776000 18446744073709552000 2.168404344971009e-19 -2
-2 -9223372036854776000 -9223372036854776000 9223372036854776000 18446744073709552000 2.168404344971009e-19 -2
-2 -140737488355329 -140737488355331 140737488355327 281474976710658 1.4210854715201903e-14 -2
-2 -140737488355328 -140737488355330 140737488355326 281474976710656 1.4210854715202004e-14 -2
-2 140737488355327 140737488355325 -140737488355329 -281474976710654 -1.4210854715202105e-14 -2
-2 140737488355328 140737488355326 -140737488355330 -281474976710656 -1.4210854715202004e-14 -2
-2 9223372036854776000 9223372036854776000 -9223372036854776000 -18446744073709552000 -2.168404344971009e-19 -2
-2 9223372036854776000 9223372036854776000 -9223372036854776000 -18446744073709552000 -2.168404344971009e-19 -2
-2 18446744073709552000 18446744073709552000 -18446744073709552000 -36893488147419103000 -1.0842021724855044e-19 -2
-2 -4294967295 -4294967297 4294967293 8589934590 4.656612874161595e-10 -2
-2 -2147483649 -2147483651 2147483647 4294967298 9.313225741817976e-10 -2
-2 -2147483648 -2147483650 2147483646 4294967296 9.313225746154785e-10 -2
-2 -2147483647 -2147483649 2147483645 4294967294 9.313225750491594e-10 -2
-2 2147483647 2147483645 -2147483649 -4294967294 -9.313225750491594e-10 -2
-2 2147483648 2147483646 -2147483650 -4294967296 -9.313225746154785e-10 -2
-2 2147483649 2147483647 -2147483651 -4294967298 -9.313225741817976e-10 -2
-2 4294967295 4294967293 -4294967297 -8589934590 -4.656612874161595e-10 -2
-2 -3 -5 1 6 0.6666666666666666 -2
-2 -2 -4 +0 4 1 -0
-2 -1 -3 -1 2 2 -0
-2 -0 -2 -2 +0 Infinity NaN
-2 +0 -2 -2 -0 -Infinity NaN
-2 1 -1 -3 -2 -2 -0
-2 2 +0 -4 -4 -1 -0
-2 3 1 -5 -6 -0.6666666666666666 -2
-1 -18446744073709552000 -18446744073709552000 18446744073709552000 18446744073709552000 5.421010862427522e-20 -1
-1 -9223372036854776000 -9223372036854776000 9223372036854776000 9223372036854776000 1.0842021724855044e-19 -1
-1 -9223372036854776000 -9223372036854776000 9223372036854776000 9223372036854776000 1.0842021724855044e-19 -1
-1 -140737488355329 -140737488355330 140737488355328 140737488355329 7.105427357600951e-15 -1
-1 -140737488355328 -140737488355329 140737488355327 140737488355328 7.105427357601002e-15 -1
-1 140737488355327 140737488355326 -140737488355328 -140737488355327 -7.105427357601052e-15 -1
-1 140737488355328 140737488355327 -140737488355329 -140737488355328 -7.105427357601002e-15 -1
-1 9223372036854776000 9223372036854776000 -9223372036854776000 -9223372036854776000 -1.0842021724855044e-19 -1
-1 9223372036854776000 9223372036854776000 -9223372036854776000 -9223372036854776000 -1.0842021724855044e-19 -1
-1 18446744073709552000 18446744073709552000 -18446744073709552000 -18446744073709552000 -5.421010862427522e-20 -1
-1 -4294967295 -4294967296 4294967294 4294967295 2.3283064370807974e-10 -1
-1 -2147483649 -2147483650 2147483648 2147483649 4.656612870908988e-10 -1
-1 -2147483648 -2147483649 2147483647 2147483648 4.656612873077393e-10 -1
-1 -2147483647 -2147483648 2147483646 2147483647 4.656612875245797e-10 -1
-1 2147483647 2147483646 -2147483648 -2147483647 -4.656612875245797e-10 -1
-1 2147483648 2147483647 -2147483649 -2147483648 -4.656612873077393e-10 -1
-1 2147483649 2147483648 -2147483650 -2147483649 -4.656612870908988e-10 -1
-1 4294967295 4294967294 -4294967296 -4294967295 -2.3283064370807974e-10 -1
-1 -3 -4 2 3 0.3333333333333333 -1
-1 -2 -3 1 2 0.5 -1
-1 -1 -2 +0 1 1 -0
-1 -0 -1 -1 +0 Infinity NaN
-1 +0 -1 -1 -0 -Infinity NaN
-1 1 +0 -2 -1 -1 -0
-1 2 1 -3 -2 -0.5 -1
-1 3 2 -4 -3 -0.3333333333333333 -1
-0 -18446744073709552000 -18446744073709552000 18446744073709552000 +0 +0 -0
-0 -9223372036854776000 -9223372036854776000 9223372036854776000 +0 +0 -0
-0 -9223372036854776000 -9223372036854776000 9223372036854776000 +0 +0 -0
-0 -140737488355329 -140737488355329 140737488355329 +0 +0 -0
-0 -140737488355328 -140737488355328 140737488355328 +0 +0 -0
-0 140737488355327 140737488355327 -140737488355327 -0 -0 -0
-0 140737488355328 140737488355328 -140737488355328 -0 -0 -0
-0 9223372036854776000 9223372036854776000 -9223372036854776000 -0 -0 -0
-0 9223372036854776000 9223372036854776000 -9223372036854776000 -0 -0 -0
-0 18446744073709552000 18446744073709552000 -18446744073709552000 -0 -0 -0
-0 -4294967295 -4294967295 4294967295 +0 +0 -0
-0 -2147483649 -2147483649 2147483649 +0 +0 -0
-0 -2147483648 -2147483648 2147483648 +0 +0 -0
-0 -2147483647 -2147483647 2147483647 +0 +0 -0
-0 2147483647 2147483647 -2147483647 -0 -0 -0
-0 2147483648 2147483648 -2147483648 -0 -0 -0
-0 2147483649 2147483649 -2147483649 -0 -0 -0
-0 4294967295 4294967295 -4294967295 -0 -0 -0
-0 -3 -3 3 +0 +0 -0
-0 -2 -2 2 +0 +0 -0
-0 -1 -1 1 +0 +0 -0
-0 -0 -0 +0 +0 NaN NaN
-0 +0 +0 -0 -0 NaN NaN
-0 1 1 -1 -0 -0 -0
-0 2 2 -2 -0 -0 -0
-0 3 3 -3 -0 -0 -0
+0 -18446744073709552000 -18446744073709552000 18446744073709552000 -0 -0 +0
+0 -9223372036854776000 -9223372036854776000 9223372036854776000 -0 -0 +0
+0 -9223372036854776000 -9223372036854776000 9223372036854776000 -0 -0 +0
+0 -140737488355329 -140737488355329 140737488355329 -0 -0 +0
+0 -140737488355328 -140737488355328 140737488355328 -0 -0 +0
+0 140737488355327 140737488355327 -140737488355327 +0 +0 +0
+0 140737488355328 140737488355328 -140737488355328 +0 +0 +0
+0 9223372036854776000 9223372036854776000 -9223372036854776000 +0 +0 +0
+0 9223372036854776000 9223372036854776000 -9223372036854776000 +0 +0 +0
+0 18446744073709552000 18446744073709552000 -18446744073709552000 +0 +0 +0
+0 -4294967295 -4294967295 4294967295 -0 -0 +0
+0 -2147483649 -2147483649 2147483649 -0 -0 +0
+0 -2147483648 -2147483648 2147483648 -0 -0 +0
+0 -2147483647 -2147483647 2147483647 -0 -0 +0
+0 2147483647 2147483647 -2147483647 +0 +0 +0
+0 2147483648 2147483648 -2147483648 +0 +0 +0
+0 2147483649 2147483649 -2147483649 +0 +0 +0
+0 4294967295 4294967295 -4294967295 +0 +0 +0
+0 -3 -3 3 -0 -0 +0
+0 -2 -2 2 -0 -0 +0
+0 -1 -1 1 -0 -0 +0
+0 -0 +0 +0 -0 NaN NaN
+0 +0 +0 +0 +0 NaN NaN
+0 1 1 -1 +0 +0 +0
+0 2 2 -2 +0 +0 +0
+0 3 3 -3 +0 +0 +0
1 -18446744073709552000 -18446744073709552000 18446744073709552000 -18446744073709552000 -5.421010862427522e-20 1
1 -9223372036854776000 -9223372036854776000 9223372036854776000 -9223372036854776000 -1.0842021724855044e-19 1
1 -9223372036854776000 -9223372036854776000 9223372036854776000 -9223372036854776000 -1.0842021724855044e-19 1
1 -140737488355329 -140737488355328 140737488355330 -140737488355329 -7.105427357600951e-15 1
1 -140737488355328 -140737488355327 140737488355329 -140737488355328 -7.105427357601002e-15 1
1 140737488355327 140737488355328 -140737488355326 140737488355327 7.105427357601052e-15 1
1 140737488355328 140737488355329 -140737488355327 140737488355328 7.105427357601002e-15 1
1 9223372036854776000 9223372036854776000 -9223372036854776000 9223372036854776000 1.0842021724855044e-19 1
1 9223372036854776000 9223372036854776000 -9223372036854776000 9223372036854776000 1.0842021724855044e-19 1
1 18446744073709552000 18446744073709552000 -18446744073709552000 18446744073709552000 5.421010862427522e-20 1
1 -4294967295 -4294967294 4294967296 -4294967295 -2.3283064370807974e-10 1
1 -2147483649 -2147483648 2147483650 -2147483649 -4.656612870908988e-10 1
1 -2147483648 -2147483647 2147483649 -2147483648 -4.656612873077393e-10 1
1 -2147483647 -2147483646 2147483648 -2147483647 -4.656612875245797e-10 1
1 2147483647 2147483648 -2147483646 2147483647 4.656612875245797e-10 1
1 2147483648 2147483649 -2147483647 2147483648 4.656612873077393e-10 1
1 2147483649 2147483650 -2147483648 2147483649 4.656612870908988e-10 1
1 4294967295 4294967296 -4294967294 4294967295 2.3283064370807974e-10 1
1 -3 -2 4 -3 -0.3333333333333333 1
1 -2 -1 3 -2 -0.5 1
1 -1 +0 2 -1 -1 +0
1 -0 1 1 -0 -Infinity NaN
1 +0 1 1 +0 Infinity NaN
1 1 2 +0 1 1 +0
1 2 3 -1 2 0.5 1
1 3 4 -2 3 0.3333333333333333 1
2 -18446744073709552000 -18446744073709552000 18446744073709552000 -36893488147419103000 -1.0842021724855044e-19 2
2 -9223372036854776000 -9223372036854776000 9223372036854776000 -18446744073709552000 -2.168404344971009e-19 2
2 -9223372036854776000 -9223372036854776000 9223372036854776000 -18446744073709552000 -2.168404344971009e-19 2
2 -140737488355329 -140737488355327 140737488355331 -281474976710658 -1.4210854715201903e-14 2
2 -140737488355328 -140737488355326 140737488355330 -281474976710656 -1.4210854715202004e-14 2
2 140737488355327 140737488355329 -140737488355325 281474976710654 1.4210854715202105e-14 2
2 140737488355328 140737488355330 -140737488355326 281474976710656 1.4210854715202004e-14 2
2 9223372036854776000 9223372036854776000 -9223372036854776000 18446744073709552000 2.168404344971009e-19 2
2 9223372036854776000 9223372036854776000 -9223372036854776000 18446744073709552000 2.168404344971009e-19 2
2 18446744073709552000 18446744073709552000 -18446744073709552000 36893488147419103000 1.0842021724855044e-19 2
2 -4294967295 -4294967293 4294967297 -8589934590 -4.656612874161595e-10 2
2 -2147483649 -2147483647 2147483651 -4294967298 -9.313225741817976e-10 2
2 -2147483648 -2147483646 2147483650 -4294967296 -9.313225746154785e-10 2
2 -2147483647 -2147483645 2147483649 -4294967294 -9.313225750491594e-10 2
2 2147483647 2147483649 -2147483645 4294967294 9.313225750491594e-10 2
2 2147483648 2147483650 -2147483646 4294967296 9.313225746154785e-10 2
2 2147483649 2147483651 -2147483647 4294967298 9.313225741817976e-10 2
2 4294967295 4294967297 -4294967293 8589934590 4.656612874161595e-10 2
2 -3 -1 5 -6 -0.6666666666666666 2
2 -2 +0 4 -4 -1 +0
2 -1 1 3 -2 -2 +0
2 -0 2 2 -0 -Infinity NaN
2 +0 2 2 +0 Infinity NaN
2 1 3 1 2 2 +0
2 2 4 +0 4 1 +0
2 3 5 -1 6 0.6666666666666666 2
3 -18446744073709552000 -18446744073709552000 18446744073709552000 -55340232221128655000 -1.6263032587282567e-19 3
3 -9223372036854776000 -9223372036854776000 9223372036854776000 -27670116110564327000 -3.2526065174565133e-19 3
3 -9223372036854776000 -9223372036854776000 9223372036854776000 -27670116110564327000 -3.2526065174565133e-19 3
3 -140737488355329 -140737488355326 140737488355332 -422212465065987 -2.1316282072802854e-14 3
3 -140737488355328 -140737488355325 140737488355331 -422212465065984 -2.1316282072803006e-14 3
3 140737488355327 140737488355330 -140737488355324 422212465065981 2.1316282072803157e-14 3
3 140737488355328 140737488355331 -140737488355325 422212465065984 2.1316282072803006e-14 3
3 9223372036854776000 9223372036854776000 -9223372036854776000 27670116110564327000 3.2526065174565133e-19 3
3 9223372036854776000 9223372036854776000 -9223372036854776000 27670116110564327000 3.2526065174565133e-19 3
3 18446744073709552000 18446744073709552000 -18446744073709552000 55340232221128655000 1.6263032587282567e-19 3
3 -4294967295 -4294967292 4294967298 -12884901885 -6.984919311242392e-10 3
3 -2147483649 -2147483646 2147483652 -6442450947 -1.3969838612726965e-9 3
3 -2147483648 -2147483645 2147483651 -6442450944 -1.3969838619232178e-9 3
3 -2147483647 -2147483644 2147483650 -6442450941 -1.396983862573739e-9 3
3 2147483647 2147483650 -2147483644 6442450941 1.396983862573739e-9 3
3 2147483648 2147483651 -2147483645 6442450944 1.3969838619232178e-9 3
3 2147483649 2147483652 -2147483646 6442450947 1.3969838612726965e-9 3
3 4294967295 4294967298 -4294967292 12884901885 6.984919311242392e-10 3
3 -3 +0 6 -9 -1 +0
3 -2 1 5 -6 -1.5 1
3 -1 2 4 -3 -3 +0
3 -0 3 3 -0 -Infinity NaN
3 +0 3 3 +0 Infinity NaN
3 1 4 2 3 3 +0
3 2 5 1 6 1.5 1
3 3 6 +0 9 1 +0
===*/

function binaryArithmeticBrute() {
    var vals = TEST_VALUES;

    print('binary brute force');

    vals.forEach(function (x) {
        vals.forEach(function (y) {
            printFixed(x, y, x + y, x - y, x * y, x / y, x % y);
        });
    });
}

/*===
binary bitops brute force
-18446744073709552000 -18446744073709552000 +0 +0 +0 +0 +0 +0
-18446744073709552000 -9223372036854776000 +0 +0 +0 +0 +0 +0
-18446744073709552000 -9223372036854776000 +0 +0 +0 +0 +0 +0
-18446744073709552000 -140737488355329 +0 -1 -1 +0 +0 +0
-18446744073709552000 -140737488355328 +0 +0 +0 +0 +0 +0
-18446744073709552000 140737488355327 +0 -1 -1 +0 +0 +0
-18446744073709552000 140737488355328 +0 +0 +0 +0 +0 +0
-18446744073709552000 9223372036854776000 +0 +0 +0 +0 +0 +0
-18446744073709552000 9223372036854776000 +0 +0 +0 +0 +0 +0
-18446744073709552000 18446744073709552000 +0 +0 +0 +0 +0 +0
-18446744073709552000 -4294967295 +0 1 1 +0 +0 +0
-18446744073709552000 -2147483649 +0 2147483647 2147483647 +0 +0 +0
-18446744073709552000 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-18446744073709552000 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
-18446744073709552000 2147483647 +0 2147483647 2147483647 +0 +0 +0
-18446744073709552000 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-18446744073709552000 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
-18446744073709552000 4294967295 +0 -1 -1 +0 +0 +0
-18446744073709552000 -3 +0 -3 -3 +0 +0 +0
-18446744073709552000 -2 +0 -2 -2 +0 +0 +0
-18446744073709552000 -1 +0 -1 -1 +0 +0 +0
-18446744073709552000 -0 +0 +0 +0 +0 +0 +0
-18446744073709552000 +0 +0 +0 +0 +0 +0 +0
-18446744073709552000 1 +0 1 1 +0 +0 +0
-18446744073709552000 2 +0 2 2 +0 +0 +0
-18446744073709552000 3 +0 3 3 +0 +0 +0
-9223372036854776000 -18446744073709552000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -140737488355329 +0 -1 -1 +0 +0 +0
-9223372036854776000 -140737488355328 +0 +0 +0 +0 +0 +0
-9223372036854776000 140737488355327 +0 -1 -1 +0 +0 +0
-9223372036854776000 140737488355328 +0 +0 +0 +0 +0 +0
-9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 18446744073709552000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -4294967295 +0 1 1 +0 +0 +0
-9223372036854776000 -2147483649 +0 2147483647 2147483647 +0 +0 +0
-9223372036854776000 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-9223372036854776000 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
-9223372036854776000 2147483647 +0 2147483647 2147483647 +0 +0 +0
-9223372036854776000 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-9223372036854776000 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
-9223372036854776000 4294967295 +0 -1 -1 +0 +0 +0
-9223372036854776000 -3 +0 -3 -3 +0 +0 +0
-9223372036854776000 -2 +0 -2 -2 +0 +0 +0
-9223372036854776000 -1 +0 -1 -1 +0 +0 +0
-9223372036854776000 -0 +0 +0 +0 +0 +0 +0
-9223372036854776000 +0 +0 +0 +0 +0 +0 +0
-9223372036854776000 1 +0 1 1 +0 +0 +0
-9223372036854776000 2 +0 2 2 +0 +0 +0
-9223372036854776000 3 +0 3 3 +0 +0 +0
-9223372036854776000 -18446744073709552000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -140737488355329 +0 -1 -1 +0 +0 +0
-9223372036854776000 -140737488355328 +0 +0 +0 +0 +0 +0
-9223372036854776000 140737488355327 +0 -1 -1 +0 +0 +0
-9223372036854776000 140737488355328 +0 +0 +0 +0 +0 +0
-9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
-9223372036854776000 18446744073709552000 +0 +0 +0 +0 +0 +0
-9223372036854776000 -4294967295 +0 1 1 +0 +0 +0
-9223372036854776000 -2147483649 +0 2147483647 2147483647 +0 +0 +0
-9223372036854776000 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-9223372036854776000 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
-9223372036854776000 2147483647 +0 2147483647 2147483647 +0 +0 +0
-9223372036854776000 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-9223372036854776000 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
-9223372036854776000 4294967295 +0 -1 -1 +0 +0 +0
-9223372036854776000 -3 +0 -3 -3 +0 +0 +0
-9223372036854776000 -2 +0 -2 -2 +0 +0 +0
-9223372036854776000 -1 +0 -1 -1 +0 +0 +0
-9223372036854776000 -0 +0 +0 +0 +0 +0 +0
-9223372036854776000 +0 +0 +0 +0 +0 +0 +0
-9223372036854776000 1 +0 1 1 +0 +0 +0
-9223372036854776000 2 +0 2 2 +0 +0 +0
-9223372036854776000 3 +0 3 3 +0 +0 +0
-140737488355329 -18446744073709552000 +0 -1 -1 -1 -1 4294967295
-140737488355329 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
-140737488355329 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
-140737488355329 -140737488355329 -1 -1 +0 -2147483648 -1 1
-140737488355329 -140737488355328 +0 -1 -1 -1 -1 4294967295
-140737488355329 140737488355327 -1 -1 +0 -2147483648 -1 1
-140737488355329 140737488355328 +0 -1 -1 -1 -1 4294967295
-140737488355329 9223372036854776000 +0 -1 -1 -1 -1 4294967295
-140737488355329 9223372036854776000 +0 -1 -1 -1 -1 4294967295
-140737488355329 18446744073709552000 +0 -1 -1 -1 -1 4294967295
-140737488355329 -4294967295 1 -1 -2 -2 -1 2147483647
-140737488355329 -2147483649 2147483647 -1 -2147483648 -2147483648 -1 1
-140737488355329 -2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
-140737488355329 -2147483647 -2147483647 -1 2147483646 -2 -1 2147483647
-140737488355329 2147483647 2147483647 -1 -2147483648 -2147483648 -1 1
-140737488355329 2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
-140737488355329 2147483649 -2147483647 -1 2147483646 -2 -1 2147483647
-140737488355329 4294967295 -1 -1 +0 -2147483648 -1 1
-140737488355329 -3 -3 -1 2 -536870912 -1 7
-140737488355329 -2 -2 -1 1 -1073741824 -1 3
-140737488355329 -1 -1 -1 +0 -2147483648 -1 1
-140737488355329 -0 +0 -1 -1 -1 -1 4294967295
-140737488355329 +0 +0 -1 -1 -1 -1 4294967295
-140737488355329 1 1 -1 -2 -2 -1 2147483647
-140737488355329 2 2 -1 -3 -4 -1 1073741823
-140737488355329 3 3 -1 -4 -8 -1 536870911
-140737488355328 -18446744073709552000 +0 +0 +0 +0 +0 +0
-140737488355328 -9223372036854776000 +0 +0 +0 +0 +0 +0
-140737488355328 -9223372036854776000 +0 +0 +0 +0 +0 +0
-140737488355328 -140737488355329 +0 -1 -1 +0 +0 +0
-140737488355328 -140737488355328 +0 +0 +0 +0 +0 +0
-140737488355328 140737488355327 +0 -1 -1 +0 +0 +0
-140737488355328 140737488355328 +0 +0 +0 +0 +0 +0
-140737488355328 9223372036854776000 +0 +0 +0 +0 +0 +0
-140737488355328 9223372036854776000 +0 +0 +0 +0 +0 +0
-140737488355328 18446744073709552000 +0 +0 +0 +0 +0 +0
-140737488355328 -4294967295 +0 1 1 +0 +0 +0
-140737488355328 -2147483649 +0 2147483647 2147483647 +0 +0 +0
-140737488355328 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-140737488355328 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
-140737488355328 2147483647 +0 2147483647 2147483647 +0 +0 +0
-140737488355328 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-140737488355328 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
-140737488355328 4294967295 +0 -1 -1 +0 +0 +0
-140737488355328 -3 +0 -3 -3 +0 +0 +0
-140737488355328 -2 +0 -2 -2 +0 +0 +0
-140737488355328 -1 +0 -1 -1 +0 +0 +0
-140737488355328 -0 +0 +0 +0 +0 +0 +0
-140737488355328 +0 +0 +0 +0 +0 +0 +0
-140737488355328 1 +0 1 1 +0 +0 +0
-140737488355328 2 +0 2 2 +0 +0 +0
-140737488355328 3 +0 3 3 +0 +0 +0
140737488355327 -18446744073709552000 +0 -1 -1 -1 -1 4294967295
140737488355327 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
140737488355327 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
140737488355327 -140737488355329 -1 -1 +0 -2147483648 -1 1
140737488355327 -140737488355328 +0 -1 -1 -1 -1 4294967295
140737488355327 140737488355327 -1 -1 +0 -2147483648 -1 1
140737488355327 140737488355328 +0 -1 -1 -1 -1 4294967295
140737488355327 9223372036854776000 +0 -1 -1 -1 -1 4294967295
140737488355327 9223372036854776000 +0 -1 -1 -1 -1 4294967295
140737488355327 18446744073709552000 +0 -1 -1 -1 -1 4294967295
140737488355327 -4294967295 1 -1 -2 -2 -1 2147483647
140737488355327 -2147483649 2147483647 -1 -2147483648 -2147483648 -1 1
140737488355327 -2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
140737488355327 -2147483647 -2147483647 -1 2147483646 -2 -1 2147483647
140737488355327 2147483647 2147483647 -1 -2147483648 -2147483648 -1 1
140737488355327 2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
140737488355327 2147483649 -2147483647 -1 2147483646 -2 -1 2147483647
140737488355327 4294967295 -1 -1 +0 -2147483648 -1 1
140737488355327 -3 -3 -1 2 -536870912 -1 7
140737488355327 -2 -2 -1 1 -1073741824 -1 3
140737488355327 -1 -1 -1 +0 -2147483648 -1 1
140737488355327 -0 +0 -1 -1 -1 -1 4294967295
140737488355327 +0 +0 -1 -1 -1 -1 4294967295
140737488355327 1 1 -1 -2 -2 -1 2147483647
140737488355327 2 2 -1 -3 -4 -1 1073741823
140737488355327 3 3 -1 -4 -8 -1 536870911
140737488355328 -18446744073709552000 +0 +0 +0 +0 +0 +0
140737488355328 -9223372036854776000 +0 +0 +0 +0 +0 +0
140737488355328 -9223372036854776000 +0 +0 +0 +0 +0 +0
140737488355328 -140737488355329 +0 -1 -1 +0 +0 +0
140737488355328 -140737488355328 +0 +0 +0 +0 +0 +0
140737488355328 140737488355327 +0 -1 -1 +0 +0 +0
140737488355328 140737488355328 +0 +0 +0 +0 +0 +0
140737488355328 9223372036854776000 +0 +0 +0 +0 +0 +0
140737488355328 9223372036854776000 +0 +0 +0 +0 +0 +0
140737488355328 18446744073709552000 +0 +0 +0 +0 +0 +0
140737488355328 -4294967295 +0 1 1 +0 +0 +0
140737488355328 -2147483649 +0 2147483647 2147483647 +0 +0 +0
140737488355328 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
140737488355328 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
140737488355328 2147483647 +0 2147483647 2147483647 +0 +0 +0
140737488355328 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
140737488355328 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
140737488355328 4294967295 +0 -1 -1 +0 +0 +0
140737488355328 -3 +0 -3 -3 +0 +0 +0
140737488355328 -2 +0 -2 -2 +0 +0 +0
140737488355328 -1 +0 -1 -1 +0 +0 +0
140737488355328 -0 +0 +0 +0 +0 +0 +0
140737488355328 +0 +0 +0 +0 +0 +0 +0
140737488355328 1 +0 1 1 +0 +0 +0
140737488355328 2 +0 2 2 +0 +0 +0
140737488355328 3 +0 3 3 +0 +0 +0
9223372036854776000 -18446744073709552000 +0 +0 +0 +0 +0 +0
9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 -140737488355329 +0 -1 -1 +0 +0 +0
9223372036854776000 -140737488355328 +0 +0 +0 +0 +0 +0
9223372036854776000 140737488355327 +0 -1 -1 +0 +0 +0
9223372036854776000 140737488355328 +0 +0 +0 +0 +0 +0
9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 18446744073709552000 +0 +0 +0 +0 +0 +0
9223372036854776000 -4294967295 +0 1 1 +0 +0 +0
9223372036854776000 -2147483649 +0 2147483647 2147483647 +0 +0 +0
9223372036854776000 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
9223372036854776000 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
9223372036854776000 2147483647 +0 2147483647 2147483647 +0 +0 +0
9223372036854776000 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
9223372036854776000 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
9223372036854776000 4294967295 +0 -1 -1 +0 +0 +0
9223372036854776000 -3 +0 -3 -3 +0 +0 +0
9223372036854776000 -2 +0 -2 -2 +0 +0 +0
9223372036854776000 -1 +0 -1 -1 +0 +0 +0
9223372036854776000 -0 +0 +0 +0 +0 +0 +0
9223372036854776000 +0 +0 +0 +0 +0 +0 +0
9223372036854776000 1 +0 1 1 +0 +0 +0
9223372036854776000 2 +0 2 2 +0 +0 +0
9223372036854776000 3 +0 3 3 +0 +0 +0
9223372036854776000 -18446744073709552000 +0 +0 +0 +0 +0 +0
9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 -9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 -140737488355329 +0 -1 -1 +0 +0 +0
9223372036854776000 -140737488355328 +0 +0 +0 +0 +0 +0
9223372036854776000 140737488355327 +0 -1 -1 +0 +0 +0
9223372036854776000 140737488355328 +0 +0 +0 +0 +0 +0
9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 9223372036854776000 +0 +0 +0 +0 +0 +0
9223372036854776000 18446744073709552000 +0 +0 +0 +0 +0 +0
9223372036854776000 -4294967295 +0 1 1 +0 +0 +0
9223372036854776000 -2147483649 +0 2147483647 2147483647 +0 +0 +0
9223372036854776000 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
9223372036854776000 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
9223372036854776000 2147483647 +0 2147483647 2147483647 +0 +0 +0
9223372036854776000 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
9223372036854776000 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
9223372036854776000 4294967295 +0 -1 -1 +0 +0 +0
9223372036854776000 -3 +0 -3 -3 +0 +0 +0
9223372036854776000 -2 +0 -2 -2 +0 +0 +0
9223372036854776000 -1 +0 -1 -1 +0 +0 +0
9223372036854776000 -0 +0 +0 +0 +0 +0 +0
9223372036854776000 +0 +0 +0 +0 +0 +0 +0
9223372036854776000 1 +0 1 1 +0 +0 +0
9223372036854776000 2 +0 2 2 +0 +0 +0
9223372036854776000 3 +0 3 3 +0 +0 +0
18446744073709552000 -18446744073709552000 +0 +0 +0 +0 +0 +0
18446744073709552000 -9223372036854776000 +0 +0 +0 +0 +0 +0
18446744073709552000 -9223372036854776000 +0 +0 +0 +0 +0 +0
18446744073709552000 -140737488355329 +0 -1 -1 +0 +0 +0
18446744073709552000 -140737488355328 +0 +0 +0 +0 +0 +0
18446744073709552000 140737488355327 +0 -1 -1 +0 +0 +0
18446744073709552000 140737488355328 +0 +0 +0 +0 +0 +0
18446744073709552000 9223372036854776000 +0 +0 +0 +0 +0 +0
18446744073709552000 9223372036854776000 +0 +0 +0 +0 +0 +0
18446744073709552000 18446744073709552000 +0 +0 +0 +0 +0 +0
18446744073709552000 -4294967295 +0 1 1 +0 +0 +0
18446744073709552000 -2147483649 +0 2147483647 2147483647 +0 +0 +0
18446744073709552000 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
18446744073709552000 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
18446744073709552000 2147483647 +0 2147483647 2147483647 +0 +0 +0
18446744073709552000 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
18446744073709552000 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
18446744073709552000 4294967295 +0 -1 -1 +0 +0 +0
18446744073709552000 -3 +0 -3 -3 +0 +0 +0
18446744073709552000 -2 +0 -2 -2 +0 +0 +0
18446744073709552000 -1 +0 -1 -1 +0 +0 +0
18446744073709552000 -0 +0 +0 +0 +0 +0 +0
18446744073709552000 +0 +0 +0 +0 +0 +0 +0
18446744073709552000 1 +0 1 1 +0 +0 +0
18446744073709552000 2 +0 2 2 +0 +0 +0
18446744073709552000 3 +0 3 3 +0 +0 +0
-4294967295 -18446744073709552000 +0 1 1 1 1 1
-4294967295 -9223372036854776000 +0 1 1 1 1 1
-4294967295 -9223372036854776000 +0 1 1 1 1 1
-4294967295 -140737488355329 1 -1 -2 -2147483648 +0 +0
-4294967295 -140737488355328 +0 1 1 1 1 1
-4294967295 140737488355327 1 -1 -2 -2147483648 +0 +0
-4294967295 140737488355328 +0 1 1 1 1 1
-4294967295 9223372036854776000 +0 1 1 1 1 1
-4294967295 9223372036854776000 +0 1 1 1 1 1
-4294967295 18446744073709552000 +0 1 1 1 1 1
-4294967295 -4294967295 1 1 +0 2 +0 +0
-4294967295 -2147483649 1 2147483647 2147483646 -2147483648 +0 +0
-4294967295 -2147483648 +0 -2147483647 -2147483647 1 1 1
-4294967295 -2147483647 1 -2147483647 -2147483648 2 +0 +0
-4294967295 2147483647 1 2147483647 2147483646 -2147483648 +0 +0
-4294967295 2147483648 +0 -2147483647 -2147483647 1 1 1
-4294967295 2147483649 1 -2147483647 -2147483648 2 +0 +0
-4294967295 4294967295 1 -1 -2 -2147483648 +0 +0
-4294967295 -3 1 -3 -4 536870912 +0 +0
-4294967295 -2 +0 -1 -1 1073741824 +0 +0
-4294967295 -1 1 -1 -2 -2147483648 +0 +0
-4294967295 -0 +0 1 1 1 1 1
-4294967295 +0 +0 1 1 1 1 1
-4294967295 1 1 1 +0 2 +0 +0
-4294967295 2 +0 3 3 4 +0 +0
-4294967295 3 1 3 2 8 +0 +0
-2147483649 -18446744073709552000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 -9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 -9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 -140737488355329 2147483647 -1 -2147483648 -2147483648 +0 +0
-2147483649 -140737488355328 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 140737488355327 2147483647 -1 -2147483648 -2147483648 +0 +0
-2147483649 140737488355328 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 18446744073709552000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 -4294967295 1 2147483647 2147483646 -2 1073741823 1073741823
-2147483649 -2147483649 2147483647 2147483647 +0 -2147483648 +0 +0
-2147483649 -2147483648 +0 -1 -1 2147483647 2147483647 2147483647
-2147483649 -2147483647 1 -1 -2 -2 1073741823 1073741823
-2147483649 2147483647 2147483647 2147483647 +0 -2147483648 +0 +0
-2147483649 2147483648 +0 -1 -1 2147483647 2147483647 2147483647
-2147483649 2147483649 1 -1 -2 -2 1073741823 1073741823
-2147483649 4294967295 2147483647 -1 -2147483648 -2147483648 +0 +0
-2147483649 -3 2147483645 -1 -2147483646 -536870912 3 3
-2147483649 -2 2147483646 -1 -2147483647 -1073741824 1 1
-2147483649 -1 2147483647 -1 -2147483648 -2147483648 +0 +0
-2147483649 -0 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 +0 +0 2147483647 2147483647 2147483647 2147483647 2147483647
-2147483649 1 1 2147483647 2147483646 -2 1073741823 1073741823
-2147483649 2 2 2147483647 2147483645 -4 536870911 536870911
-2147483649 3 3 2147483647 2147483644 -8 268435455 268435455
-2147483648 -18446744073709552000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 -9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 -9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 -140737488355329 -2147483648 -1 2147483647 +0 -1 1
-2147483648 -140737488355328 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 140737488355327 -2147483648 -1 2147483647 +0 -1 1
-2147483648 140737488355328 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 18446744073709552000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 -4294967295 +0 -2147483647 -2147483647 +0 -1073741824 1073741824
-2147483648 -2147483649 +0 -1 -1 +0 -1 1
-2147483648 -2147483648 -2147483648 -2147483648 +0 -2147483648 -2147483648 2147483648
-2147483648 -2147483647 -2147483648 -2147483647 1 +0 -1073741824 1073741824
-2147483648 2147483647 +0 -1 -1 +0 -1 1
-2147483648 2147483648 -2147483648 -2147483648 +0 -2147483648 -2147483648 2147483648
-2147483648 2147483649 -2147483648 -2147483647 1 +0 -1073741824 1073741824
-2147483648 4294967295 -2147483648 -1 2147483647 +0 -1 1
-2147483648 -3 -2147483648 -3 2147483645 +0 -4 4
-2147483648 -2 -2147483648 -2 2147483646 +0 -2 2
-2147483648 -1 -2147483648 -1 2147483647 +0 -1 1
-2147483648 -0 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 +0 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
-2147483648 1 +0 -2147483647 -2147483647 +0 -1073741824 1073741824
-2147483648 2 +0 -2147483646 -2147483646 +0 -536870912 536870912
-2147483648 3 +0 -2147483645 -2147483645 +0 -268435456 268435456
-2147483647 -18446744073709552000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 -9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 -9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 -140737488355329 -2147483647 -1 2147483646 -2147483648 -1 1
-2147483647 -140737488355328 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 140737488355327 -2147483647 -1 2147483646 -2147483648 -1 1
-2147483647 140737488355328 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 18446744073709552000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 -4294967295 1 -2147483647 -2147483648 2 -1073741824 1073741824
-2147483647 -2147483649 1 -1 -2 -2147483648 -1 1
-2147483647 -2147483648 -2147483648 -2147483647 1 -2147483647 -2147483647 2147483649
-2147483647 -2147483647 -2147483647 -2147483647 +0 2 -1073741824 1073741824
-2147483647 2147483647 1 -1 -2 -2147483648 -1 1
-2147483647 2147483648 -2147483648 -2147483647 1 -2147483647 -2147483647 2147483649
-2147483647 2147483649 -2147483647 -2147483647 +0 2 -1073741824 1073741824
-2147483647 4294967295 -2147483647 -1 2147483646 -2147483648 -1 1
-2147483647 -3 -2147483647 -3 2147483644 536870912 -4 4
-2147483647 -2 -2147483648 -1 2147483647 1073741824 -2 2
-2147483647 -1 -2147483647 -1 2147483646 -2147483648 -1 1
-2147483647 -0 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 +0 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
-2147483647 1 1 -2147483647 -2147483648 2 -1073741824 1073741824
-2147483647 2 +0 -2147483645 -2147483645 4 -536870912 536870912
-2147483647 3 1 -2147483645 -2147483646 8 -268435456 268435456
2147483647 -18446744073709552000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 -9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 -9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 -140737488355329 2147483647 -1 -2147483648 -2147483648 +0 +0
2147483647 -140737488355328 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 140737488355327 2147483647 -1 -2147483648 -2147483648 +0 +0
2147483647 140737488355328 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 9223372036854776000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 18446744073709552000 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 -4294967295 1 2147483647 2147483646 -2 1073741823 1073741823
2147483647 -2147483649 2147483647 2147483647 +0 -2147483648 +0 +0
2147483647 -2147483648 +0 -1 -1 2147483647 2147483647 2147483647
2147483647 -2147483647 1 -1 -2 -2 1073741823 1073741823
2147483647 2147483647 2147483647 2147483647 +0 -2147483648 +0 +0
2147483647 2147483648 +0 -1 -1 2147483647 2147483647 2147483647
2147483647 2147483649 1 -1 -2 -2 1073741823 1073741823
2147483647 4294967295 2147483647 -1 -2147483648 -2147483648 +0 +0
2147483647 -3 2147483645 -1 -2147483646 -536870912 3 3
2147483647 -2 2147483646 -1 -2147483647 -1073741824 1 1
2147483647 -1 2147483647 -1 -2147483648 -2147483648 +0 +0
2147483647 -0 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 +0 +0 2147483647 2147483647 2147483647 2147483647 2147483647
2147483647 1 1 2147483647 2147483646 -2 1073741823 1073741823
2147483647 2 2 2147483647 2147483645 -4 536870911 536870911
2147483647 3 3 2147483647 2147483644 -8 268435455 268435455
2147483648 -18446744073709552000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 -9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 -9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 -140737488355329 -2147483648 -1 2147483647 +0 -1 1
2147483648 -140737488355328 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 140737488355327 -2147483648 -1 2147483647 +0 -1 1
2147483648 140737488355328 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 9223372036854776000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 18446744073709552000 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 -4294967295 +0 -2147483647 -2147483647 +0 -1073741824 1073741824
2147483648 -2147483649 +0 -1 -1 +0 -1 1
2147483648 -2147483648 -2147483648 -2147483648 +0 -2147483648 -2147483648 2147483648
2147483648 -2147483647 -2147483648 -2147483647 1 +0 -1073741824 1073741824
2147483648 2147483647 +0 -1 -1 +0 -1 1
2147483648 2147483648 -2147483648 -2147483648 +0 -2147483648 -2147483648 2147483648
2147483648 2147483649 -2147483648 -2147483647 1 +0 -1073741824 1073741824
2147483648 4294967295 -2147483648 -1 2147483647 +0 -1 1
2147483648 -3 -2147483648 -3 2147483645 +0 -4 4
2147483648 -2 -2147483648 -2 2147483646 +0 -2 2
2147483648 -1 -2147483648 -1 2147483647 +0 -1 1
2147483648 -0 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 +0 +0 -2147483648 -2147483648 -2147483648 -2147483648 2147483648
2147483648 1 +0 -2147483647 -2147483647 +0 -1073741824 1073741824
2147483648 2 +0 -2147483646 -2147483646 +0 -536870912 536870912
2147483648 3 +0 -2147483645 -2147483645 +0 -268435456 268435456
2147483649 -18446744073709552000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 -9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 -9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 -140737488355329 -2147483647 -1 2147483646 -2147483648 -1 1
2147483649 -140737488355328 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 140737488355327 -2147483647 -1 2147483646 -2147483648 -1 1
2147483649 140737488355328 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 9223372036854776000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 18446744073709552000 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 -4294967295 1 -2147483647 -2147483648 2 -1073741824 1073741824
2147483649 -2147483649 1 -1 -2 -2147483648 -1 1
2147483649 -2147483648 -2147483648 -2147483647 1 -2147483647 -2147483647 2147483649
2147483649 -2147483647 -2147483647 -2147483647 +0 2 -1073741824 1073741824
2147483649 2147483647 1 -1 -2 -2147483648 -1 1
2147483649 2147483648 -2147483648 -2147483647 1 -2147483647 -2147483647 2147483649
2147483649 2147483649 -2147483647 -2147483647 +0 2 -1073741824 1073741824
2147483649 4294967295 -2147483647 -1 2147483646 -2147483648 -1 1
2147483649 -3 -2147483647 -3 2147483644 536870912 -4 4
2147483649 -2 -2147483648 -1 2147483647 1073741824 -2 2
2147483649 -1 -2147483647 -1 2147483646 -2147483648 -1 1
2147483649 -0 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 +0 +0 -2147483647 -2147483647 -2147483647 -2147483647 2147483649
2147483649 1 1 -2147483647 -2147483648 2 -1073741824 1073741824
2147483649 2 +0 -2147483645 -2147483645 4 -536870912 536870912
2147483649 3 1 -2147483645 -2147483646 8 -268435456 268435456
4294967295 -18446744073709552000 +0 -1 -1 -1 -1 4294967295
4294967295 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
4294967295 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
4294967295 -140737488355329 -1 -1 +0 -2147483648 -1 1
4294967295 -140737488355328 +0 -1 -1 -1 -1 4294967295
4294967295 140737488355327 -1 -1 +0 -2147483648 -1 1
4294967295 140737488355328 +0 -1 -1 -1 -1 4294967295
4294967295 9223372036854776000 +0 -1 -1 -1 -1 4294967295
4294967295 9223372036854776000 +0 -1 -1 -1 -1 4294967295
4294967295 18446744073709552000 +0 -1 -1 -1 -1 4294967295
4294967295 -4294967295 1 -1 -2 -2 -1 2147483647
4294967295 -2147483649 2147483647 -1 -2147483648 -2147483648 -1 1
4294967295 -2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
4294967295 -2147483647 -2147483647 -1 2147483646 -2 -1 2147483647
4294967295 2147483647 2147483647 -1 -2147483648 -2147483648 -1 1
4294967295 2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
4294967295 2147483649 -2147483647 -1 2147483646 -2 -1 2147483647
4294967295 4294967295 -1 -1 +0 -2147483648 -1 1
4294967295 -3 -3 -1 2 -536870912 -1 7
4294967295 -2 -2 -1 1 -1073741824 -1 3
4294967295 -1 -1 -1 +0 -2147483648 -1 1
4294967295 -0 +0 -1 -1 -1 -1 4294967295
4294967295 +0 +0 -1 -1 -1 -1 4294967295
4294967295 1 1 -1 -2 -2 -1 2147483647
4294967295 2 2 -1 -3 -4 -1 1073741823
4294967295 3 3 -1 -4 -8 -1 536870911
-3 -18446744073709552000 +0 -3 -3 -3 -3 4294967293
-3 -9223372036854776000 +0 -3 -3 -3 -3 4294967293
-3 -9223372036854776000 +0 -3 -3 -3 -3 4294967293
-3 -140737488355329 -3 -1 2 -2147483648 -1 1
-3 -140737488355328 +0 -3 -3 -3 -3 4294967293
-3 140737488355327 -3 -1 2 -2147483648 -1 1
-3 140737488355328 +0 -3 -3 -3 -3 4294967293
-3 9223372036854776000 +0 -3 -3 -3 -3 4294967293
-3 9223372036854776000 +0 -3 -3 -3 -3 4294967293
-3 18446744073709552000 +0 -3 -3 -3 -3 4294967293
-3 -4294967295 1 -3 -4 -6 -2 2147483646
-3 -2147483649 2147483645 -1 -2147483646 -2147483648 -1 1
-3 -2147483648 -2147483648 -3 2147483645 -3 -3 4294967293
-3 -2147483647 -2147483647 -3 2147483644 -6 -2 2147483646
-3 2147483647 2147483645 -1 -2147483646 -2147483648 -1 1
-3 2147483648 -2147483648 -3 2147483645 -3 -3 4294967293
-3 2147483649 -2147483647 -3 2147483644 -6 -2 2147483646
-3 4294967295 -3 -1 2 -2147483648 -1 1
-3 -3 -3 -3 +0 -1610612736 -1 7
-3 -2 -4 -1 3 1073741824 -1 3
-3 -1 -3 -1 2 -2147483648 -1 1
-3 -0 +0 -3 -3 -3 -3 4294967293
-3 +0 +0 -3 -3 -3 -3 4294967293
-3 1 1 -3 -4 -6 -2 2147483646
-3 2 +0 -1 -1 -12 -1 1073741823
-3 3 1 -1 -2 -24 -1 536870911
-2 -18446744073709552000 +0 -2 -2 -2 -2 4294967294
-2 -9223372036854776000 +0 -2 -2 -2 -2 4294967294
-2 -9223372036854776000 +0 -2 -2 -2 -2 4294967294
-2 -140737488355329 -2 -1 1 +0 -1 1
-2 -140737488355328 +0 -2 -2 -2 -2 4294967294
-2 140737488355327 -2 -1 1 +0 -1 1
-2 140737488355328 +0 -2 -2 -2 -2 4294967294
-2 9223372036854776000 +0 -2 -2 -2 -2 4294967294
-2 9223372036854776000 +0 -2 -2 -2 -2 4294967294
-2 18446744073709552000 +0 -2 -2 -2 -2 4294967294
-2 -4294967295 +0 -1 -1 -4 -1 2147483647
-2 -2147483649 2147483646 -1 -2147483647 +0 -1 1
-2 -2147483648 -2147483648 -2 2147483646 -2 -2 4294967294
-2 -2147483647 -2147483648 -1 2147483647 -4 -1 2147483647
-2 2147483647 2147483646 -1 -2147483647 +0 -1 1
-2 2147483648 -2147483648 -2 2147483646 -2 -2 4294967294
-2 2147483649 -2147483648 -1 2147483647 -4 -1 2147483647
-2 4294967295 -2 -1 1 +0 -1 1
-2 -3 -4 -1 3 -1073741824 -1 7
-2 -2 -2 -2 +0 -2147483648 -1 3
-2 -1 -2 -1 1 +0 -1 1
-2 -0 +0 -2 -2 -2 -2 4294967294
-2 +0 +0 -2 -2 -2 -2 4294967294
-2 1 +0 -1 -1 -4 -1 2147483647
-2 2 2 -2 -4 -8 -1 1073741823
-2 3 2 -1 -3 -16 -1 536870911
-1 -18446744073709552000 +0 -1 -1 -1 -1 4294967295
-1 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
-1 -9223372036854776000 +0 -1 -1 -1 -1 4294967295
-1 -140737488355329 -1 -1 +0 -2147483648 -1 1
-1 -140737488355328 +0 -1 -1 -1 -1 4294967295
-1 140737488355327 -1 -1 +0 -2147483648 -1 1
-1 140737488355328 +0 -1 -1 -1 -1 4294967295
-1 9223372036854776000 +0 -1 -1 -1 -1 4294967295
-1 9223372036854776000 +0 -1 -1 -1 -1 4294967295
-1 18446744073709552000 +0 -1 -1 -1 -1 4294967295
-1 -4294967295 1 -1 -2 -2 -1 2147483647
-1 -2147483649 2147483647 -1 -2147483648 -2147483648 -1 1
-1 -2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
-1 -2147483647 -2147483647 -1 2147483646 -2 -1 2147483647
-1 2147483647 2147483647 -1 -2147483648 -2147483648 -1 1
-1 2147483648 -2147483648 -1 2147483647 -1 -1 4294967295
-1 2147483649 -2147483647 -1 2147483646 -2 -1 2147483647
-1 4294967295 -1 -1 +0 -2147483648 -1 1
-1 -3 -3 -1 2 -536870912 -1 7
-1 -2 -2 -1 1 -1073741824 -1 3
-1 -1 -1 -1 +0 -2147483648 -1 1
-1 -0 +0 -1 -1 -1 -1 4294967295
-1 +0 +0 -1 -1 -1 -1 4294967295
-1 1 1 -1 -2 -2 -1 2147483647
-1 2 2 -1 -3 -4 -1 1073741823
-1 3 3 -1 -4 -8 -1 536870911
-0 -18446744073709552000 +0 +0 +0 +0 +0 +0
-0 -9223372036854776000 +0 +0 +0 +0 +0 +0
-0 -9223372036854776000 +0 +0 +0 +0 +0 +0
-0 -140737488355329 +0 -1 -1 +0 +0 +0
-0 -140737488355328 +0 +0 +0 +0 +0 +0
-0 140737488355327 +0 -1 -1 +0 +0 +0
-0 140737488355328 +0 +0 +0 +0 +0 +0
-0 9223372036854776000 +0 +0 +0 +0 +0 +0
-0 9223372036854776000 +0 +0 +0 +0 +0 +0
-0 18446744073709552000 +0 +0 +0 +0 +0 +0
-0 -4294967295 +0 1 1 +0 +0 +0
-0 -2147483649 +0 2147483647 2147483647 +0 +0 +0
-0 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-0 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
-0 2147483647 +0 2147483647 2147483647 +0 +0 +0
-0 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
-0 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
-0 4294967295 +0 -1 -1 +0 +0 +0
-0 -3 +0 -3 -3 +0 +0 +0
-0 -2 +0 -2 -2 +0 +0 +0
-0 -1 +0 -1 -1 +0 +0 +0
-0 -0 +0 +0 +0 +0 +0 +0
-0 +0 +0 +0 +0 +0 +0 +0
-0 1 +0 1 1 +0 +0 +0
-0 2 +0 2 2 +0 +0 +0
-0 3 +0 3 3 +0 +0 +0
+0 -18446744073709552000 +0 +0 +0 +0 +0 +0
+0 -9223372036854776000 +0 +0 +0 +0 +0 +0
+0 -9223372036854776000 +0 +0 +0 +0 +0 +0
+0 -140737488355329 +0 -1 -1 +0 +0 +0
+0 -140737488355328 +0 +0 +0 +0 +0 +0
+0 140737488355327 +0 -1 -1 +0 +0 +0
+0 140737488355328 +0 +0 +0 +0 +0 +0
+0 9223372036854776000 +0 +0 +0 +0 +0 +0
+0 9223372036854776000 +0 +0 +0 +0 +0 +0
+0 18446744073709552000 +0 +0 +0 +0 +0 +0
+0 -4294967295 +0 1 1 +0 +0 +0
+0 -2147483649 +0 2147483647 2147483647 +0 +0 +0
+0 -2147483648 +0 -2147483648 -2147483648 +0 +0 +0
+0 -2147483647 +0 -2147483647 -2147483647 +0 +0 +0
+0 2147483647 +0 2147483647 2147483647 +0 +0 +0
+0 2147483648 +0 -2147483648 -2147483648 +0 +0 +0
+0 2147483649 +0 -2147483647 -2147483647 +0 +0 +0
+0 4294967295 +0 -1 -1 +0 +0 +0
+0 -3 +0 -3 -3 +0 +0 +0
+0 -2 +0 -2 -2 +0 +0 +0
+0 -1 +0 -1 -1 +0 +0 +0
+0 -0 +0 +0 +0 +0 +0 +0
+0 +0 +0 +0 +0 +0 +0 +0
+0 1 +0 1 1 +0 +0 +0
+0 2 +0 2 2 +0 +0 +0
+0 3 +0 3 3 +0 +0 +0
1 -18446744073709552000 +0 1 1 1 1 1
1 -9223372036854776000 +0 1 1 1 1 1
1 -9223372036854776000 +0 1 1 1 1 1
1 -140737488355329 1 -1 -2 -2147483648 +0 +0
1 -140737488355328 +0 1 1 1 1 1
1 140737488355327 1 -1 -2 -2147483648 +0 +0
1 140737488355328 +0 1 1 1 1 1
1 9223372036854776000 +0 1 1 1 1 1
1 9223372036854776000 +0 1 1 1 1 1
1 18446744073709552000 +0 1 1 1 1 1
1 -4294967295 1 1 +0 2 +0 +0
1 -2147483649 1 2147483647 2147483646 -2147483648 +0 +0
1 -2147483648 +0 -2147483647 -2147483647 1 1 1
1 -2147483647 1 -2147483647 -2147483648 2 +0 +0
1 2147483647 1 2147483647 2147483646 -2147483648 +0 +0
1 2147483648 +0 -2147483647 -2147483647 1 1 1
1 2147483649 1 -2147483647 -2147483648 2 +0 +0
1 4294967295 1 -1 -2 -2147483648 +0 +0
1 -3 1 -3 -4 536870912 +0 +0
1 -2 +0 -1 -1 1073741824 +0 +0
1 -1 1 -1 -2 -2147483648 +0 +0
1 -0 +0 1 1 1 1 1
1 +0 +0 1 1 1 1 1
1 1 1 1 +0 2 +0 +0
1 2 +0 3 3 4 +0 +0
1 3 1 3 2 8 +0 +0
2 -18446744073709552000 +0 2 2 2 2 2
2 -9223372036854776000 +0 2 2 2 2 2
2 -9223372036854776000 +0 2 2 2 2 2
2 -140737488355329 2 -1 -3 +0 +0 +0
2 -140737488355328 +0 2 2 2 2 2
2 140737488355327 2 -1 -3 +0 +0 +0
2 140737488355328 +0 2 2 2 2 2
2 9223372036854776000 +0 2 2 2 2 2
2 9223372036854776000 +0 2 2 2 2 2
2 18446744073709552000 +0 2 2 2 2 2
2 -4294967295 +0 3 3 4 1 1
2 -2147483649 2 2147483647 2147483645 +0 +0 +0
2 -2147483648 +0 -2147483646 -2147483646 2 2 2
2 -2147483647 +0 -2147483645 -2147483645 4 1 1
2 2147483647 2 2147483647 2147483645 +0 +0 +0
2 2147483648 +0 -2147483646 -2147483646 2 2 2
2 2147483649 +0 -2147483645 -2147483645 4 1 1
2 4294967295 2 -1 -3 +0 +0 +0
2 -3 +0 -1 -1 1073741824 +0 +0
2 -2 2 -2 -4 -2147483648 +0 +0
2 -1 2 -1 -3 +0 +0 +0
2 -0 +0 2 2 2 2 2
2 +0 +0 2 2 2 2 2
2 1 +0 3 3 4 1 1
2 2 2 2 +0 8 +0 +0
2 3 2 3 1 16 +0 +0
3 -18446744073709552000 +0 3 3 3 3 3
3 -9223372036854776000 +0 3 3 3 3 3
3 -9223372036854776000 +0 3 3 3 3 3
3 -140737488355329 3 -1 -4 -2147483648 +0 +0
3 -140737488355328 +0 3 3 3 3 3
3 140737488355327 3 -1 -4 -2147483648 +0 +0
3 140737488355328 +0 3 3 3 3 3
3 9223372036854776000 +0 3 3 3 3 3
3 9223372036854776000 +0 3 3 3 3 3
3 18446744073709552000 +0 3 3 3 3 3
3 -4294967295 1 3 2 6 1 1
3 -2147483649 3 2147483647 2147483644 -2147483648 +0 +0
3 -2147483648 +0 -2147483645 -2147483645 3 3 3
3 -2147483647 1 -2147483645 -2147483646 6 1 1
3 2147483647 3 2147483647 2147483644 -2147483648 +0 +0
3 2147483648 +0 -2147483645 -2147483645 3 3 3
3 2147483649 1 -2147483645 -2147483646 6 1 1
3 4294967295 3 -1 -4 -2147483648 +0 +0
3 -3 1 -1 -2 1610612736 +0 +0
3 -2 2 -1 -3 -1073741824 +0 +0
3 -1 3 -1 -4 -2147483648 +0 +0
3 -0 +0 3 3 3 3 3
3 +0 +0 3 3 3 3 3
3 1 1 3 2 6 1 1
3 2 2 3 1 12 +0 +0
3 3 3 3 +0 24 +0 +0
===*/

function binaryBitopsBrute() {
    var vals = TEST_VALUES;

    print('binary bitops brute force');

    vals.forEach(function (x) {
        vals.forEach(function (y) {
            printFixed(x, y, x & y, x | y, x ^ y, x << y, x >> y, x >>> y);
        });
    });
}

/*===
unary arith in range
140737488355327 7fffffffffff fastint
-140737488355328 -800000000000 fastint
-140737488355327 -7fffffffffff fastint
140737488355327 7fffffffffff fastint
+0 0 fastint
-0 0
unary out of range
-0 0
+0 0
===*/

function unaryArithmeticTest() {
    var x, y;

    print('unary arith in range');
    x = MIN_FASTINT_PLUS1; y = -x; printFastint(y);
    x = MIN_FASTINT; y = +x; printFastint(y);
    x = MAX_FASTINT; y = -x; printFastint(y);
    x = MAX_FASTINT; y = +x; printFastint(y);
    x = POS_ZERO; y = +x; printFastint(y);
    x = NEG_ZERO; y = +x; printFastint(y);

    print('unary out of range');
    x = POS_ZERO; y = -x; printFastint(y);
    x = NEG_ZERO; y = -x; printFastint(y);
}

/*===
unary brute force
-18446744073709552000 -18446744073709552000 18446744073709552000 false
-9223372036854776000 -9223372036854776000 9223372036854776000 false
-9223372036854776000 -9223372036854776000 9223372036854776000 false
-140737488355329 -140737488355329 140737488355329 false
-140737488355328 -140737488355328 140737488355328 false
140737488355327 140737488355327 -140737488355327 false
140737488355328 140737488355328 -140737488355328 false
9223372036854776000 9223372036854776000 -9223372036854776000 false
9223372036854776000 9223372036854776000 -9223372036854776000 false
18446744073709552000 18446744073709552000 -18446744073709552000 false
-4294967295 -4294967295 4294967295 false
-2147483649 -2147483649 2147483649 false
-2147483648 -2147483648 2147483648 false
-2147483647 -2147483647 2147483647 false
2147483647 2147483647 -2147483647 false
2147483648 2147483648 -2147483648 false
2147483649 2147483649 -2147483649 false
4294967295 4294967295 -4294967295 false
-3 -3 3 false
-2 -2 2 false
-1 -1 1 false
-0 -0 +0 true
+0 +0 -0 true
1 1 -1 false
2 2 -2 false
3 3 -3 false
===*/

function unaryArithmeticBrute() {
    var vals = TEST_VALUES;

    print('unary brute force');

    vals.forEach(function (x) {
        printFixed(x, +x, -x, !x);
    });
}

/*===
unary bitops brute force
-18446744073709552000 -1
-9223372036854776000 -1
-9223372036854776000 -1
-140737488355329 +0
-140737488355328 -1
140737488355327 +0
140737488355328 -1
9223372036854776000 -1
9223372036854776000 -1
18446744073709552000 -1
-4294967295 -2
-2147483649 -2147483648
-2147483648 2147483647
-2147483647 2147483646
2147483647 -2147483648
2147483648 2147483647
2147483649 2147483646
4294967295 +0
-3 2
-2 1
-1 +0
-0 -1
+0 -1
1 -2
2 -3
3 -4
===*/

function unaryBitopsBrute() {
    var vals = TEST_VALUES;

    print('unary bitops brute force');

    vals.forEach(function (x) {
        printFixed(x, ~x);
    });
}

/*===
negative zero
-0 0
+0 0
+0 0 fastint
===*/

function negativeZeroTest() {
    print('negative zero');

    x = NEG_ZERO;
    printFastint(x);

    x = NEG_ZERO;
    y = 0;
    z = x + y;
    printFastint(z);  // not yet fastint
    z = +z;
    printFastint(z);
}

/*===
return value downgrade test
123 7b fastint
111 6f fastint
123 7b fastint
===*/

function retvalDowngradeTest() {
    // All function return values (both ECMAScript and C) are automatically
    // double-to-fastint downgraded.

    function myfunc() {
        var x = 123.0;
        x += 0.5;
        return x - 0.5;
    }

    print('return value downgrade test');
    printFastint(Math.floor(123.1));
    printFastint('foo'.charCodeAt(1));
    printFastint(myfunc());
}

/*===
yield/resume value downgrade test
x before initial resume
123 7b
arg in thread
123 7b fastint
x before initial yield
123 7b
yield value
123 7b fastint
===*/

function yieldResumeDowngradeTest() {
    // All yielded values are automatically double-to-fastint downgraded.
    // All resume values are automatically double-to-fastint downgraded.

    print('yield/resume value downgrade test');

    function myThread(arg) {
        print('arg in thread');
        printFastint(arg);

        var x = 123.0;
        x += 0.5; x -= 0.5;
        print('x before initial yield');
        printFastint(x);
        var resumeValue = Duktape.Thread.yield(x);

        print('resume value');
        printFastint(resumeValue);
    }

    var t = new Duktape.Thread(myThread);

    var x = 123.0;
    x += 0.5; x -= 0.5;
    print('x before initial resume');
    printFastint(x);
    var yieldValue = Duktape.Thread.resume(t, x);
    print('yield value');
    printFastint(yieldValue);
}

/*===
unary plus downgrade test
123 7b fastint
123.1 7b.199999999998
123 7b
123 7b fastint
123 7b fastint
===*/

function unaryPlusDowngradeTest() {
     var x;

     print('unary plus downgrade test');

     x = 123;
     printFastint(x);
     x += 0.1;
     printFastint(x);
     x -= 0.1;  // fastint compatible but not auto-downgraded
     printFastint(x);
     x = +x;    // downgraded by unary plus
     printFastint(x);

     // This gets constant folded and downgraded as a compiler constant.

     x = 123 + 0.1 - 0.1;
     printFastint(x);
}

/*===
downgrade sanity
+0 0
+0 0 fastint
-0 0
-0 0
-1.0000000000000002 -1.0000000000001
-1.0000000000000002 -1.0000000000001
-2.220446049250313e-16 -0.0000000000001
-2.220446049250313e-16 -0.0000000000001
0.9999999999999998 0.fffffffffffff
0.9999999999999998 0.fffffffffffff
-0.9999999999999998 -0.fffffffffffff
-0.9999999999999998 -0.fffffffffffff
2.220446049250313e-16 0.0000000000001
2.220446049250313e-16 0.0000000000001
1.0000000000000002 1.0000000000001
1.0000000000000002 1.0000000000001
-1.0000000000000004 -1.0000000000002
-1.0000000000000004 -1.0000000000002
-4.440892098500626e-16 -0.0000000000002
-4.440892098500626e-16 -0.0000000000002
0.9999999999999996 0.ffffffffffffe
0.9999999999999996 0.ffffffffffffe
-0.9999999999999996 -0.ffffffffffffe
-0.9999999999999996 -0.ffffffffffffe
4.440892098500626e-16 0.0000000000002
4.440892098500626e-16 0.0000000000002
1.0000000000000004 1.0000000000002
1.0000000000000004 1.0000000000002
-1.0000000000000009 -1.0000000000004
-1.0000000000000009 -1.0000000000004
-8.881784197001252e-16 -0.0000000000004
-8.881784197001252e-16 -0.0000000000004
0.9999999999999991 0.ffffffffffffc
0.9999999999999991 0.ffffffffffffc
-0.9999999999999991 -0.ffffffffffffc
-0.9999999999999991 -0.ffffffffffffc
8.881784197001252e-16 0.0000000000004
8.881784197001252e-16 0.0000000000004
1.0000000000000009 1.0000000000004
1.0000000000000009 1.0000000000004
-1.0000000000000018 -1.0000000000008
-1.0000000000000018 -1.0000000000008
-1.7763568394002505e-15 -0.0000000000008
-1.7763568394002505e-15 -0.0000000000008
0.9999999999999982 0.ffffffffffff8
0.9999999999999982 0.ffffffffffff8
-0.9999999999999982 -0.ffffffffffff8
-0.9999999999999982 -0.ffffffffffff8
1.7763568394002505e-15 0.0000000000008
1.7763568394002505e-15 0.0000000000008
1.0000000000000018 1.0000000000008
1.0000000000000018 1.0000000000008
-1.0000000000000036 -1.000000000001
-1.0000000000000036 -1.000000000001
-3.552713678800501e-15 -0.000000000001
-3.552713678800501e-15 -0.000000000001
0.9999999999999964 0.ffffffffffff
0.9999999999999964 0.ffffffffffff
-0.9999999999999964 -0.ffffffffffff
-0.9999999999999964 -0.ffffffffffff
3.552713678800501e-15 0.000000000001
3.552713678800501e-15 0.000000000001
1.0000000000000036 1.000000000001
1.0000000000000036 1.000000000001
-1.000000000000007 -1.000000000002
-1.000000000000007 -1.000000000002
-7.105427357601002e-15 -0.000000000002
-7.105427357601002e-15 -0.000000000002
0.9999999999999929 0.fffffffffffe
0.9999999999999929 0.fffffffffffe
-0.9999999999999929 -0.fffffffffffe
-0.9999999999999929 -0.fffffffffffe
7.105427357601002e-15 0.000000000002
7.105427357601002e-15 0.000000000002
1.000000000000007 1.000000000002
1.000000000000007 1.000000000002
-1.0000000000000142 -1.000000000004
-1.0000000000000142 -1.000000000004
-1.4210854715202004e-14 -0.000000000004
-1.4210854715202004e-14 -0.000000000004
0.9999999999999858 0.fffffffffffc
0.9999999999999858 0.fffffffffffc
-0.9999999999999858 -0.fffffffffffc
-0.9999999999999858 -0.fffffffffffc
1.4210854715202004e-14 0.000000000004
1.4210854715202004e-14 0.000000000004
1.0000000000000142 1.000000000004
1.0000000000000142 1.000000000004
-1.0000000000000284 -1.000000000008
-1.0000000000000284 -1.000000000008
-2.842170943040401e-14 -0.000000000008
-2.842170943040401e-14 -0.000000000008
0.9999999999999716 0.fffffffffff8
0.9999999999999716 0.fffffffffff8
-0.9999999999999716 -0.fffffffffff8
-0.9999999999999716 -0.fffffffffff8
2.842170943040401e-14 0.000000000008
2.842170943040401e-14 0.000000000008
1.0000000000000284 1.000000000008
1.0000000000000284 1.000000000008
-1.0000000000000568 -1.00000000001
-1.0000000000000568 -1.00000000001
-5.684341886080802e-14 -0.00000000001
-5.684341886080802e-14 -0.00000000001
0.9999999999999432 0.fffffffffff
0.9999999999999432 0.fffffffffff
-0.9999999999999432 -0.fffffffffff
-0.9999999999999432 -0.fffffffffff
5.684341886080802e-14 0.00000000001
5.684341886080802e-14 0.00000000001
1.0000000000000568 1.00000000001
1.0000000000000568 1.00000000001
-1.0000000000001137 -1.00000000002
-1.0000000000001137 -1.00000000002
-1.1368683772161603e-13 -0.00000000002
-1.1368683772161603e-13 -0.00000000002
0.9999999999998863 0.ffffffffffe
0.9999999999998863 0.ffffffffffe
-0.9999999999998863 -0.ffffffffffe
-0.9999999999998863 -0.ffffffffffe
1.1368683772161603e-13 0.00000000002
1.1368683772161603e-13 0.00000000002
1.0000000000001137 1.00000000002
1.0000000000001137 1.00000000002
-1.0000000000002274 -1.00000000004
-1.0000000000002274 -1.00000000004
-2.2737367544323206e-13 -0.00000000004
-2.2737367544323206e-13 -0.00000000004
0.9999999999997726 0.ffffffffffc
0.9999999999997726 0.ffffffffffc
-0.9999999999997726 -0.ffffffffffc
-0.9999999999997726 -0.ffffffffffc
2.2737367544323206e-13 0.00000000004
2.2737367544323206e-13 0.00000000004
1.0000000000002274 1.00000000004
1.0000000000002274 1.00000000004
-1.0000000000004547 -1.00000000008
-1.0000000000004547 -1.00000000008
-4.547473508864641e-13 -0.00000000008
-4.547473508864641e-13 -0.00000000008
0.9999999999995453 0.ffffffffff8
0.9999999999995453 0.ffffffffff8
-0.9999999999995453 -0.ffffffffff8
-0.9999999999995453 -0.ffffffffff8
4.547473508864641e-13 0.00000000008
4.547473508864641e-13 0.00000000008
1.0000000000004547 1.00000000008
1.0000000000004547 1.00000000008
-1.0000000000009095 -1.0000000001
-1.0000000000009095 -1.0000000001
-9.094947017729282e-13 -0.0000000001
-9.094947017729282e-13 -0.0000000001
0.9999999999990905 0.ffffffffff
0.9999999999990905 0.ffffffffff
-0.9999999999990905 -0.ffffffffff
-0.9999999999990905 -0.ffffffffff
9.094947017729282e-13 0.0000000001
9.094947017729282e-13 0.0000000001
1.0000000000009095 1.0000000001
1.0000000000009095 1.0000000001
-1.000000000001819 -1.0000000002
-1.000000000001819 -1.0000000002
-1.8189894035458565e-12 -0.0000000002
-1.8189894035458565e-12 -0.0000000002
0.999999999998181 0.fffffffffe
0.999999999998181 0.fffffffffe
-0.999999999998181 -0.fffffffffe
-0.999999999998181 -0.fffffffffe
1.8189894035458565e-12 0.0000000002
1.8189894035458565e-12 0.0000000002
1.000000000001819 1.0000000002
1.000000000001819 1.0000000002
-1.000000000003638 -1.0000000004
-1.000000000003638 -1.0000000004
-3.637978807091713e-12 -0.0000000004
-3.637978807091713e-12 -0.0000000004
0.999999999996362 0.fffffffffc
0.999999999996362 0.fffffffffc
-0.999999999996362 -0.fffffffffc
-0.999999999996362 -0.fffffffffc
3.637978807091713e-12 0.0000000004
3.637978807091713e-12 0.0000000004
1.000000000003638 1.0000000004
1.000000000003638 1.0000000004
-1.000000000007276 -1.0000000008
-1.000000000007276 -1.0000000008
-7.275957614183426e-12 -0.0000000008
-7.275957614183426e-12 -0.0000000008
0.999999999992724 0.fffffffff8
0.999999999992724 0.fffffffff8
-0.999999999992724 -0.fffffffff8
-0.999999999992724 -0.fffffffff8
7.275957614183426e-12 0.0000000008
7.275957614183426e-12 0.0000000008
1.000000000007276 1.0000000008
1.000000000007276 1.0000000008
-1.000000000014552 -1.000000001
-1.000000000014552 -1.000000001
-1.4551915228366852e-11 -0.000000001
-1.4551915228366852e-11 -0.000000001
0.9999999999854481 0.fffffffff
0.9999999999854481 0.fffffffff
-0.9999999999854481 -0.fffffffff
-0.9999999999854481 -0.fffffffff
1.4551915228366852e-11 0.000000001
1.4551915228366852e-11 0.000000001
1.000000000014552 1.000000001
1.000000000014552 1.000000001
-1.0000000000291038 -1.000000002
-1.0000000000291038 -1.000000002
-2.9103830456733704e-11 -0.000000002
-2.9103830456733704e-11 -0.000000002
0.9999999999708962 0.ffffffffe
0.9999999999708962 0.ffffffffe
-0.9999999999708962 -0.ffffffffe
-0.9999999999708962 -0.ffffffffe
2.9103830456733704e-11 0.000000002
2.9103830456733704e-11 0.000000002
1.0000000000291038 1.000000002
1.0000000000291038 1.000000002
-1.0000000000582077 -1.000000004
-1.0000000000582077 -1.000000004
-5.820766091346741e-11 -0.000000004
-5.820766091346741e-11 -0.000000004
0.9999999999417923 0.ffffffffc
0.9999999999417923 0.ffffffffc
-0.9999999999417923 -0.ffffffffc
-0.9999999999417923 -0.ffffffffc
5.820766091346741e-11 0.000000004
5.820766091346741e-11 0.000000004
1.0000000000582077 1.000000004
1.0000000000582077 1.000000004
-1.0000000001164153 -1.000000008
-1.0000000001164153 -1.000000008
-1.1641532182693481e-10 -0.000000008
-1.1641532182693481e-10 -0.000000008
0.9999999998835847 0.ffffffff8
0.9999999998835847 0.ffffffff8
-0.9999999998835847 -0.ffffffff8
-0.9999999998835847 -0.ffffffff8
1.1641532182693481e-10 0.000000008
1.1641532182693481e-10 0.000000008
1.0000000001164153 1.000000008
1.0000000001164153 1.000000008
-1.0000000002328306 -1.00000001
-1.0000000002328306 -1.00000001
-2.3283064365386963e-10 -0.00000001
-2.3283064365386963e-10 -0.00000001
0.9999999997671694 0.ffffffff
0.9999999997671694 0.ffffffff
-0.9999999997671694 -0.ffffffff
-0.9999999997671694 -0.ffffffff
2.3283064365386963e-10 0.00000001
2.3283064365386963e-10 0.00000001
1.0000000002328306 1.00000001
1.0000000002328306 1.00000001
-1.0000000004656613 -1.00000002
-1.0000000004656613 -1.00000002
-4.656612873077393e-10 -0.00000002
-4.656612873077393e-10 -0.00000002
0.9999999995343387 0.fffffffe
0.9999999995343387 0.fffffffe
-0.9999999995343387 -0.fffffffe
-0.9999999995343387 -0.fffffffe
4.656612873077393e-10 0.00000002
4.656612873077393e-10 0.00000002
1.0000000004656613 1.00000002
1.0000000004656613 1.00000002
-1.0000000009313226 -1.00000004
-1.0000000009313226 -1.00000004
-9.313225746154785e-10 -0.00000004
-9.313225746154785e-10 -0.00000004
0.9999999990686774 0.fffffffc
0.9999999990686774 0.fffffffc
-0.9999999990686774 -0.fffffffc
-0.9999999990686774 -0.fffffffc
9.313225746154785e-10 0.00000004
9.313225746154785e-10 0.00000004
1.0000000009313226 1.00000004
1.0000000009313226 1.00000004
-1.0000000018626451 -1.00000008
-1.0000000018626451 -1.00000008
-1.862645149230957e-9 -0.00000008
-1.862645149230957e-9 -0.00000008
0.9999999981373549 0.fffffff8
0.9999999981373549 0.fffffff8
-0.9999999981373549 -0.fffffff8
-0.9999999981373549 -0.fffffff8
1.862645149230957e-9 0.00000008
1.862645149230957e-9 0.00000008
1.0000000018626451 1.00000008
1.0000000018626451 1.00000008
-1.0000000037252903 -1.0000001
-1.0000000037252903 -1.0000001
-3.725290298461914e-9 -0.0000001
-3.725290298461914e-9 -0.0000001
0.9999999962747097 0.fffffff
0.9999999962747097 0.fffffff
-0.9999999962747097 -0.fffffff
-0.9999999962747097 -0.fffffff
3.725290298461914e-9 0.0000001
3.725290298461914e-9 0.0000001
1.0000000037252903 1.0000001
1.0000000037252903 1.0000001
-1.0000000074505806 -1.0000002
-1.0000000074505806 -1.0000002
-7.450580596923828e-9 -0.0000002
-7.450580596923828e-9 -0.0000002
0.9999999925494194 0.ffffffe
0.9999999925494194 0.ffffffe
-0.9999999925494194 -0.ffffffe
-0.9999999925494194 -0.ffffffe
7.450580596923828e-9 0.0000002
7.450580596923828e-9 0.0000002
1.0000000074505806 1.0000002
1.0000000074505806 1.0000002
-1.0000000149011612 -1.0000004
-1.0000000149011612 -1.0000004
-1.4901161193847656e-8 -0.0000004
-1.4901161193847656e-8 -0.0000004
0.9999999850988388 0.ffffffc
0.9999999850988388 0.ffffffc
-0.9999999850988388 -0.ffffffc
-0.9999999850988388 -0.ffffffc
1.4901161193847656e-8 0.0000004
1.4901161193847656e-8 0.0000004
1.0000000149011612 1.0000004
1.0000000149011612 1.0000004
-1.0000000298023224 -1.0000008
-1.0000000298023224 -1.0000008
-2.9802322387695313e-8 -0.0000008
-2.9802322387695313e-8 -0.0000008
0.9999999701976776 0.ffffff8
0.9999999701976776 0.ffffff8
-0.9999999701976776 -0.ffffff8
-0.9999999701976776 -0.ffffff8
2.9802322387695313e-8 0.0000008
2.9802322387695313e-8 0.0000008
1.0000000298023224 1.0000008
1.0000000298023224 1.0000008
-1.0000000596046448 -1.000001
-1.0000000596046448 -1.000001
-5.960464477539063e-8 -0.000001
-5.960464477539063e-8 -0.000001
0.9999999403953552 0.ffffff
0.9999999403953552 0.ffffff
-0.9999999403953552 -0.ffffff
-0.9999999403953552 -0.ffffff
5.960464477539063e-8 0.000001
5.960464477539063e-8 0.000001
1.0000000596046448 1.000001
1.0000000596046448 1.000001
-1.0000001192092896 -1.000002
-1.0000001192092896 -1.000002
-1.1920928955078125e-7 -0.000002
-1.1920928955078125e-7 -0.000002
0.9999998807907104 0.fffffe
0.9999998807907104 0.fffffe
-0.9999998807907104 -0.fffffe
-0.9999998807907104 -0.fffffe
1.1920928955078125e-7 0.000002
1.1920928955078125e-7 0.000002
1.0000001192092896 1.000002
1.0000001192092896 1.000002
-1.000000238418579 -1.000004
-1.000000238418579 -1.000004
-2.384185791015625e-7 -0.000004
-2.384185791015625e-7 -0.000004
0.9999997615814209 0.fffffc
0.9999997615814209 0.fffffc
-0.9999997615814209 -0.fffffc
-0.9999997615814209 -0.fffffc
2.384185791015625e-7 0.000004
2.384185791015625e-7 0.000004
1.000000238418579 1.000004
1.000000238418579 1.000004
-1.0000004768371582 -1.000008
-1.0000004768371582 -1.000008
-4.76837158203125e-7 -0.000008
-4.76837158203125e-7 -0.000008
0.9999995231628418 0.fffff8
0.9999995231628418 0.fffff8
-0.9999995231628418 -0.fffff8
-0.9999995231628418 -0.fffff8
4.76837158203125e-7 0.000008
4.76837158203125e-7 0.000008
1.0000004768371582 1.000008
1.0000004768371582 1.000008
-1.0000009536743164 -1.00001
-1.0000009536743164 -1.00001
-9.5367431640625e-7 -0.00001
-9.5367431640625e-7 -0.00001
0.9999990463256836 0.fffff
0.9999990463256836 0.fffff
-0.9999990463256836 -0.fffff
-0.9999990463256836 -0.fffff
9.5367431640625e-7 0.00001
9.5367431640625e-7 0.00001
1.0000009536743164 1.00001
1.0000009536743164 1.00001
-1.0000019073486328 -1.00002
-1.0000019073486328 -1.00002
-0.0000019073486328125 -0.00002
-0.0000019073486328125 -0.00002
0.9999980926513672 0.ffffe
0.9999980926513672 0.ffffe
-0.9999980926513672 -0.ffffe
-0.9999980926513672 -0.ffffe
0.0000019073486328125 0.00002
0.0000019073486328125 0.00002
1.0000019073486328 1.00002
1.0000019073486328 1.00002
-1.0000038146972656 -1.00004
-1.0000038146972656 -1.00004
-0.000003814697265625 -0.00004
-0.000003814697265625 -0.00004
0.9999961853027344 0.ffffc
0.9999961853027344 0.ffffc
-0.9999961853027344 -0.ffffc
-0.9999961853027344 -0.ffffc
0.000003814697265625 0.00004
0.000003814697265625 0.00004
1.0000038146972656 1.00004
1.0000038146972656 1.00004
-1.0000076293945313 -1.00008
-1.0000076293945313 -1.00008
-0.00000762939453125 -0.00008
-0.00000762939453125 -0.00008
0.9999923706054688 0.ffff8
0.9999923706054688 0.ffff8
-0.9999923706054688 -0.ffff8
-0.9999923706054688 -0.ffff8
0.00000762939453125 0.00008
0.00000762939453125 0.00008
1.0000076293945313 1.00008
1.0000076293945313 1.00008
-1.0000152587890625 -1.0001
-1.0000152587890625 -1.0001
-0.0000152587890625 -0.0001
-0.0000152587890625 -0.0001
0.9999847412109375 0.ffff
0.9999847412109375 0.ffff
-0.9999847412109375 -0.ffff
-0.9999847412109375 -0.ffff
0.0000152587890625 0.0001
0.0000152587890625 0.0001
1.0000152587890625 1.0001
1.0000152587890625 1.0001
-1.000030517578125 -1.0002
-1.000030517578125 -1.0002
-0.000030517578125 -0.0002
-0.000030517578125 -0.0002
0.999969482421875 0.fffe
0.999969482421875 0.fffe
-0.999969482421875 -0.fffe
-0.999969482421875 -0.fffe
0.000030517578125 0.0002
0.000030517578125 0.0002
1.000030517578125 1.0002
1.000030517578125 1.0002
-1.00006103515625 -1.0004
-1.00006103515625 -1.0004
-0.00006103515625 -0.0004
-0.00006103515625 -0.0004
0.99993896484375 0.fffc
0.99993896484375 0.fffc
-0.99993896484375 -0.fffc
-0.99993896484375 -0.fffc
0.00006103515625 0.0004
0.00006103515625 0.0004
1.00006103515625 1.0004
1.00006103515625 1.0004
-1.0001220703125 -1.0008
-1.0001220703125 -1.0008
-0.0001220703125 -0.0008
-0.0001220703125 -0.0008
0.9998779296875 0.fff8
0.9998779296875 0.fff8
-0.9998779296875 -0.fff8
-0.9998779296875 -0.fff8
0.0001220703125 0.0008
0.0001220703125 0.0008
1.0001220703125 1.0008
1.0001220703125 1.0008
-1.000244140625 -1.001
-1.000244140625 -1.001
-0.000244140625 -0.001
-0.000244140625 -0.001
0.999755859375 0.fff
0.999755859375 0.fff
-0.999755859375 -0.fff
-0.999755859375 -0.fff
0.000244140625 0.001
0.000244140625 0.001
1.000244140625 1.001
1.000244140625 1.001
-1.00048828125 -1.002
-1.00048828125 -1.002
-0.00048828125 -0.002
-0.00048828125 -0.002
0.99951171875 0.ffe
0.99951171875 0.ffe
-0.99951171875 -0.ffe
-0.99951171875 -0.ffe
0.00048828125 0.002
0.00048828125 0.002
1.00048828125 1.002
1.00048828125 1.002
-1.0009765625 -1.004
-1.0009765625 -1.004
-0.0009765625 -0.004
-0.0009765625 -0.004
0.9990234375 0.ffc
0.9990234375 0.ffc
-0.9990234375 -0.ffc
-0.9990234375 -0.ffc
0.0009765625 0.004
0.0009765625 0.004
1.0009765625 1.004
1.0009765625 1.004
-1.001953125 -1.008
-1.001953125 -1.008
-0.001953125 -0.008
-0.001953125 -0.008
0.998046875 0.ff8
0.998046875 0.ff8
-0.998046875 -0.ff8
-0.998046875 -0.ff8
0.001953125 0.008
0.001953125 0.008
1.001953125 1.008
1.001953125 1.008
-1.00390625 -1.01
-1.00390625 -1.01
-0.00390625 -0.01
-0.00390625 -0.01
0.99609375 0.ff
0.99609375 0.ff
-0.99609375 -0.ff
-0.99609375 -0.ff
0.00390625 0.01
0.00390625 0.01
1.00390625 1.01
1.00390625 1.01
-1.0078125 -1.02
-1.0078125 -1.02
-0.0078125 -0.02
-0.0078125 -0.02
0.9921875 0.fe
0.9921875 0.fe
-0.9921875 -0.fe
-0.9921875 -0.fe
0.0078125 0.02
0.0078125 0.02
1.0078125 1.02
1.0078125 1.02
-1.015625 -1.04
-1.015625 -1.04
-0.015625 -0.04
-0.015625 -0.04
0.984375 0.fc
0.984375 0.fc
-0.984375 -0.fc
-0.984375 -0.fc
0.015625 0.04
0.015625 0.04
1.015625 1.04
1.015625 1.04
-1.03125 -1.08
-1.03125 -1.08
-0.03125 -0.08
-0.03125 -0.08
0.96875 0.f8
0.96875 0.f8
-0.96875 -0.f8
-0.96875 -0.f8
0.03125 0.08
0.03125 0.08
1.03125 1.08
1.03125 1.08
-1.0625 -1.1
-1.0625 -1.1
-0.0625 -0.1
-0.0625 -0.1
0.9375 0.f
0.9375 0.f
-0.9375 -0.f
-0.9375 -0.f
0.0625 0.1
0.0625 0.1
1.0625 1.1
1.0625 1.1
-1.125 -1.2
-1.125 -1.2
-0.125 -0.2
-0.125 -0.2
0.875 0.e
0.875 0.e
-0.875 -0.e
-0.875 -0.e
0.125 0.2
0.125 0.2
1.125 1.2
1.125 1.2
-1.25 -1.4
-1.25 -1.4
-0.25 -0.4
-0.25 -0.4
0.75 0.c
0.75 0.c
-0.75 -0.c
-0.75 -0.c
0.25 0.4
0.25 0.4
1.25 1.4
1.25 1.4
-1.5 -1.8
-1.5 -1.8
-0.5 -0.8
-0.5 -0.8
0.5 0.8
0.5 0.8
-0.5 -0.8
-0.5 -0.8
0.5 0.8
0.5 0.8
1.5 1.8
1.5 1.8
-2 -2
-2 -2 fastint
-1 -1
-1 -1 fastint
+0 0
+0 0 fastint
+0 0
+0 0 fastint
1 1
1 1 fastint
2 2
2 2 fastint
-3 -3
-3 -3 fastint
-2 -2
-2 -2 fastint
-1 -1
-1 -1 fastint
1 1
1 1 fastint
2 2
2 2 fastint
3 3
3 3 fastint
-5 -5
-5 -5 fastint
-4 -4
-4 -4 fastint
-3 -3
-3 -3 fastint
3 3
3 3 fastint
4 4
4 4 fastint
5 5
5 5 fastint
-9 -9
-9 -9 fastint
-8 -8
-8 -8 fastint
-7 -7
-7 -7 fastint
7 7
7 7 fastint
8 8
8 8 fastint
9 9
9 9 fastint
-17 -11
-17 -11 fastint
-16 -10
-16 -10 fastint
-15 -f
-15 -f fastint
15 f
15 f fastint
16 10
16 10 fastint
17 11
17 11 fastint
-33 -21
-33 -21 fastint
-32 -20
-32 -20 fastint
-31 -1f
-31 -1f fastint
31 1f
31 1f fastint
32 20
32 20 fastint
33 21
33 21 fastint
-65 -41
-65 -41 fastint
-64 -40
-64 -40 fastint
-63 -3f
-63 -3f fastint
63 3f
63 3f fastint
64 40
64 40 fastint
65 41
65 41 fastint
-129 -81
-129 -81 fastint
-128 -80
-128 -80 fastint
-127 -7f
-127 -7f fastint
127 7f
127 7f fastint
128 80
128 80 fastint
129 81
129 81 fastint
-257 -101
-257 -101 fastint
-256 -100
-256 -100 fastint
-255 -ff
-255 -ff fastint
255 ff
255 ff fastint
256 100
256 100 fastint
257 101
257 101 fastint
-513 -201
-513 -201 fastint
-512 -200
-512 -200 fastint
-511 -1ff
-511 -1ff fastint
511 1ff
511 1ff fastint
512 200
512 200 fastint
513 201
513 201 fastint
-1025 -401
-1025 -401 fastint
-1024 -400
-1024 -400 fastint
-1023 -3ff
-1023 -3ff fastint
1023 3ff
1023 3ff fastint
1024 400
1024 400 fastint
1025 401
1025 401 fastint
-2049 -801
-2049 -801 fastint
-2048 -800
-2048 -800 fastint
-2047 -7ff
-2047 -7ff fastint
2047 7ff
2047 7ff fastint
2048 800
2048 800 fastint
2049 801
2049 801 fastint
-4097 -1001
-4097 -1001 fastint
-4096 -1000
-4096 -1000 fastint
-4095 -fff
-4095 -fff fastint
4095 fff
4095 fff fastint
4096 1000
4096 1000 fastint
4097 1001
4097 1001 fastint
-8193 -2001
-8193 -2001 fastint
-8192 -2000
-8192 -2000 fastint
-8191 -1fff
-8191 -1fff fastint
8191 1fff
8191 1fff fastint
8192 2000
8192 2000 fastint
8193 2001
8193 2001 fastint
-16385 -4001
-16385 -4001 fastint
-16384 -4000
-16384 -4000 fastint
-16383 -3fff
-16383 -3fff fastint
16383 3fff
16383 3fff fastint
16384 4000
16384 4000 fastint
16385 4001
16385 4001 fastint
-32769 -8001
-32769 -8001 fastint
-32768 -8000
-32768 -8000 fastint
-32767 -7fff
-32767 -7fff fastint
32767 7fff
32767 7fff fastint
32768 8000
32768 8000 fastint
32769 8001
32769 8001 fastint
-65537 -10001
-65537 -10001 fastint
-65536 -10000
-65536 -10000 fastint
-65535 -ffff
-65535 -ffff fastint
65535 ffff
65535 ffff fastint
65536 10000
65536 10000 fastint
65537 10001
65537 10001 fastint
-131073 -20001
-131073 -20001 fastint
-131072 -20000
-131072 -20000 fastint
-131071 -1ffff
-131071 -1ffff fastint
131071 1ffff
131071 1ffff fastint
131072 20000
131072 20000 fastint
131073 20001
131073 20001 fastint
-262145 -40001
-262145 -40001 fastint
-262144 -40000
-262144 -40000 fastint
-262143 -3ffff
-262143 -3ffff fastint
262143 3ffff
262143 3ffff fastint
262144 40000
262144 40000 fastint
262145 40001
262145 40001 fastint
-524289 -80001
-524289 -80001 fastint
-524288 -80000
-524288 -80000 fastint
-524287 -7ffff
-524287 -7ffff fastint
524287 7ffff
524287 7ffff fastint
524288 80000
524288 80000 fastint
524289 80001
524289 80001 fastint
-1048577 -100001
-1048577 -100001 fastint
-1048576 -100000
-1048576 -100000 fastint
-1048575 -fffff
-1048575 -fffff fastint
1048575 fffff
1048575 fffff fastint
1048576 100000
1048576 100000 fastint
1048577 100001
1048577 100001 fastint
-2097153 -200001
-2097153 -200001 fastint
-2097152 -200000
-2097152 -200000 fastint
-2097151 -1fffff
-2097151 -1fffff fastint
2097151 1fffff
2097151 1fffff fastint
2097152 200000
2097152 200000 fastint
2097153 200001
2097153 200001 fastint
-4194305 -400001
-4194305 -400001 fastint
-4194304 -400000
-4194304 -400000 fastint
-4194303 -3fffff
-4194303 -3fffff fastint
4194303 3fffff
4194303 3fffff fastint
4194304 400000
4194304 400000 fastint
4194305 400001
4194305 400001 fastint
-8388609 -800001
-8388609 -800001 fastint
-8388608 -800000
-8388608 -800000 fastint
-8388607 -7fffff
-8388607 -7fffff fastint
8388607 7fffff
8388607 7fffff fastint
8388608 800000
8388608 800000 fastint
8388609 800001
8388609 800001 fastint
-16777217 -1000001
-16777217 -1000001 fastint
-16777216 -1000000
-16777216 -1000000 fastint
-16777215 -ffffff
-16777215 -ffffff fastint
16777215 ffffff
16777215 ffffff fastint
16777216 1000000
16777216 1000000 fastint
16777217 1000001
16777217 1000001 fastint
-33554433 -2000001
-33554433 -2000001 fastint
-33554432 -2000000
-33554432 -2000000 fastint
-33554431 -1ffffff
-33554431 -1ffffff fastint
33554431 1ffffff
33554431 1ffffff fastint
33554432 2000000
33554432 2000000 fastint
33554433 2000001
33554433 2000001 fastint
-67108865 -4000001
-67108865 -4000001 fastint
-67108864 -4000000
-67108864 -4000000 fastint
-67108863 -3ffffff
-67108863 -3ffffff fastint
67108863 3ffffff
67108863 3ffffff fastint
67108864 4000000
67108864 4000000 fastint
67108865 4000001
67108865 4000001 fastint
-134217729 -8000001
-134217729 -8000001 fastint
-134217728 -8000000
-134217728 -8000000 fastint
-134217727 -7ffffff
-134217727 -7ffffff fastint
134217727 7ffffff
134217727 7ffffff fastint
134217728 8000000
134217728 8000000 fastint
134217729 8000001
134217729 8000001 fastint
-268435457 -10000001
-268435457 -10000001 fastint
-268435456 -10000000
-268435456 -10000000 fastint
-268435455 -fffffff
-268435455 -fffffff fastint
268435455 fffffff
268435455 fffffff fastint
268435456 10000000
268435456 10000000 fastint
268435457 10000001
268435457 10000001 fastint
-536870913 -20000001
-536870913 -20000001 fastint
-536870912 -20000000
-536870912 -20000000 fastint
-536870911 -1fffffff
-536870911 -1fffffff fastint
536870911 1fffffff
536870911 1fffffff fastint
536870912 20000000
536870912 20000000 fastint
536870913 20000001
536870913 20000001 fastint
-1073741825 -40000001
-1073741825 -40000001 fastint
-1073741824 -40000000
-1073741824 -40000000 fastint
-1073741823 -3fffffff
-1073741823 -3fffffff fastint
1073741823 3fffffff
1073741823 3fffffff fastint
1073741824 40000000
1073741824 40000000 fastint
1073741825 40000001
1073741825 40000001 fastint
-2147483649 -80000001
-2147483649 -80000001 fastint
-2147483648 -80000000
-2147483648 -80000000 fastint
-2147483647 -7fffffff
-2147483647 -7fffffff fastint
2147483647 7fffffff
2147483647 7fffffff fastint
2147483648 80000000
2147483648 80000000 fastint
2147483649 80000001
2147483649 80000001 fastint
-4294967297 -100000001
-4294967297 -100000001 fastint
-4294967296 -100000000
-4294967296 -100000000 fastint
-4294967295 -ffffffff
-4294967295 -ffffffff fastint
4294967295 ffffffff
4294967295 ffffffff fastint
4294967296 100000000
4294967296 100000000 fastint
4294967297 100000001
4294967297 100000001 fastint
-8589934593 -200000001
-8589934593 -200000001 fastint
-8589934592 -200000000
-8589934592 -200000000 fastint
-8589934591 -1ffffffff
-8589934591 -1ffffffff fastint
8589934591 1ffffffff
8589934591 1ffffffff fastint
8589934592 200000000
8589934592 200000000 fastint
8589934593 200000001
8589934593 200000001 fastint
-17179869185 -400000001
-17179869185 -400000001 fastint
-17179869184 -400000000
-17179869184 -400000000 fastint
-17179869183 -3ffffffff
-17179869183 -3ffffffff fastint
17179869183 3ffffffff
17179869183 3ffffffff fastint
17179869184 400000000
17179869184 400000000 fastint
17179869185 400000001
17179869185 400000001 fastint
-34359738369 -800000001
-34359738369 -800000001 fastint
-34359738368 -800000000
-34359738368 -800000000 fastint
-34359738367 -7ffffffff
-34359738367 -7ffffffff fastint
34359738367 7ffffffff
34359738367 7ffffffff fastint
34359738368 800000000
34359738368 800000000 fastint
34359738369 800000001
34359738369 800000001 fastint
-68719476737 -1000000001
-68719476737 -1000000001 fastint
-68719476736 -1000000000
-68719476736 -1000000000 fastint
-68719476735 -fffffffff
-68719476735 -fffffffff fastint
68719476735 fffffffff
68719476735 fffffffff fastint
68719476736 1000000000
68719476736 1000000000 fastint
68719476737 1000000001
68719476737 1000000001 fastint
-137438953473 -2000000001
-137438953473 -2000000001 fastint
-137438953472 -2000000000
-137438953472 -2000000000 fastint
-137438953471 -1fffffffff
-137438953471 -1fffffffff fastint
137438953471 1fffffffff
137438953471 1fffffffff fastint
137438953472 2000000000
137438953472 2000000000 fastint
137438953473 2000000001
137438953473 2000000001 fastint
-274877906945 -4000000001
-274877906945 -4000000001 fastint
-274877906944 -4000000000
-274877906944 -4000000000 fastint
-274877906943 -3fffffffff
-274877906943 -3fffffffff fastint
274877906943 3fffffffff
274877906943 3fffffffff fastint
274877906944 4000000000
274877906944 4000000000 fastint
274877906945 4000000001
274877906945 4000000001 fastint
-549755813889 -8000000001
-549755813889 -8000000001 fastint
-549755813888 -8000000000
-549755813888 -8000000000 fastint
-549755813887 -7fffffffff
-549755813887 -7fffffffff fastint
549755813887 7fffffffff
549755813887 7fffffffff fastint
549755813888 8000000000
549755813888 8000000000 fastint
549755813889 8000000001
549755813889 8000000001 fastint
-1099511627777 -10000000001
-1099511627777 -10000000001 fastint
-1099511627776 -10000000000
-1099511627776 -10000000000 fastint
-1099511627775 -ffffffffff
-1099511627775 -ffffffffff fastint
1099511627775 ffffffffff
1099511627775 ffffffffff fastint
1099511627776 10000000000
1099511627776 10000000000 fastint
1099511627777 10000000001
1099511627777 10000000001 fastint
-2199023255553 -20000000001
-2199023255553 -20000000001 fastint
-2199023255552 -20000000000
-2199023255552 -20000000000 fastint
-2199023255551 -1ffffffffff
-2199023255551 -1ffffffffff fastint
2199023255551 1ffffffffff
2199023255551 1ffffffffff fastint
2199023255552 20000000000
2199023255552 20000000000 fastint
2199023255553 20000000001
2199023255553 20000000001 fastint
-4398046511105 -40000000001
-4398046511105 -40000000001 fastint
-4398046511104 -40000000000
-4398046511104 -40000000000 fastint
-4398046511103 -3ffffffffff
-4398046511103 -3ffffffffff fastint
4398046511103 3ffffffffff
4398046511103 3ffffffffff fastint
4398046511104 40000000000
4398046511104 40000000000 fastint
4398046511105 40000000001
4398046511105 40000000001 fastint
-8796093022209 -80000000001
-8796093022209 -80000000001 fastint
-8796093022208 -80000000000
-8796093022208 -80000000000 fastint
-8796093022207 -7ffffffffff
-8796093022207 -7ffffffffff fastint
8796093022207 7ffffffffff
8796093022207 7ffffffffff fastint
8796093022208 80000000000
8796093022208 80000000000 fastint
8796093022209 80000000001
8796093022209 80000000001 fastint
-17592186044417 -100000000001
-17592186044417 -100000000001 fastint
-17592186044416 -100000000000
-17592186044416 -100000000000 fastint
-17592186044415 -fffffffffff
-17592186044415 -fffffffffff fastint
17592186044415 fffffffffff
17592186044415 fffffffffff fastint
17592186044416 100000000000
17592186044416 100000000000 fastint
17592186044417 100000000001
17592186044417 100000000001 fastint
-35184372088833 -200000000001
-35184372088833 -200000000001 fastint
-35184372088832 -200000000000
-35184372088832 -200000000000 fastint
-35184372088831 -1fffffffffff
-35184372088831 -1fffffffffff fastint
35184372088831 1fffffffffff
35184372088831 1fffffffffff fastint
35184372088832 200000000000
35184372088832 200000000000 fastint
35184372088833 200000000001
35184372088833 200000000001 fastint
-70368744177665 -400000000001
-70368744177665 -400000000001 fastint
-70368744177664 -400000000000
-70368744177664 -400000000000 fastint
-70368744177663 -3fffffffffff
-70368744177663 -3fffffffffff fastint
70368744177663 3fffffffffff
70368744177663 3fffffffffff fastint
70368744177664 400000000000
70368744177664 400000000000 fastint
70368744177665 400000000001
70368744177665 400000000001 fastint
-140737488355329 -800000000001
-140737488355329 -800000000001
-140737488355328 -800000000000
-140737488355328 -800000000000 fastint
-140737488355327 -7fffffffffff
-140737488355327 -7fffffffffff fastint
140737488355327 7fffffffffff
140737488355327 7fffffffffff fastint
140737488355328 800000000000
140737488355328 800000000000
140737488355329 800000000001
140737488355329 800000000001
-281474976710657 -1000000000001
-281474976710657 -1000000000001
-281474976710656 -1000000000000
-281474976710656 -1000000000000
-281474976710655 -ffffffffffff
-281474976710655 -ffffffffffff
281474976710655 ffffffffffff
281474976710655 ffffffffffff
281474976710656 1000000000000
281474976710656 1000000000000
281474976710657 1000000000001
281474976710657 1000000000001
-562949953421313 -2000000000001
-562949953421313 -2000000000001
-562949953421312 -2000000000000
-562949953421312 -2000000000000
-562949953421311 -1ffffffffffff
-562949953421311 -1ffffffffffff
562949953421311 1ffffffffffff
562949953421311 1ffffffffffff
562949953421312 2000000000000
562949953421312 2000000000000
562949953421313 2000000000001
562949953421313 2000000000001
-1125899906842625 -4000000000001
-1125899906842625 -4000000000001
-1125899906842624 -4000000000000
-1125899906842624 -4000000000000
-1125899906842623 -3ffffffffffff
-1125899906842623 -3ffffffffffff
1125899906842623 3ffffffffffff
1125899906842623 3ffffffffffff
1125899906842624 4000000000000
1125899906842624 4000000000000
1125899906842625 4000000000001
1125899906842625 4000000000001
-2251799813685249 -8000000000001
-2251799813685249 -8000000000001
-2251799813685248 -8000000000000
-2251799813685248 -8000000000000
-2251799813685247 -7ffffffffffff
-2251799813685247 -7ffffffffffff
2251799813685247 7ffffffffffff
2251799813685247 7ffffffffffff
2251799813685248 8000000000000
2251799813685248 8000000000000
2251799813685249 8000000000001
2251799813685249 8000000000001
-4503599627370496 -10000000000000
-4503599627370496 -10000000000000
-4503599627370496 -10000000000000
-4503599627370496 -10000000000000
-4503599627370495 -fffffffffffff
-4503599627370495 -fffffffffffff
4503599627370495 fffffffffffff
4503599627370495 fffffffffffff
4503599627370495.5 fffffffffffff.8
4503599627370495.5 fffffffffffff.8
4503599627370498 10000000000002
4503599627370498 10000000000002
-Infinity -Infinity
-Infinity -Infinity
Infinity Infinity
Infinity Infinity
NaN NaN
NaN NaN
===*/

function downgradeSanityTest() {
    /* The double-to-fastint downgrade check is manually written IEEE double
     * inspection and conversion, so exercise all relevant exponents.
     */

    var vals = [];
    var i, x, y;

    function twopower(n) {
        var ret = 1;
        while (n != 0) {
            if (n > 0) { ret *= 2; n--; }
            else { ret /= 2; n++; }
        }
        return ret;
    }

    print('downgrade sanity');

    for (i = -52; i <= 52; i++) {
        vals.push(-twopower(i) - 1);
        vals.push(-twopower(i));
        vals.push(-twopower(i) + 1);
        vals.push(twopower(i) - 1);
        vals.push(twopower(i));
        vals.push(twopower(i) + 1);
    }
    vals.push(Number.NEGATIVE_INFINITY);
    vals.push(Number.POSITIVE_INFINITY);
    vals.push(Number.NaN);

    // zero testing special case
    x = 1 / Number.POSITIVE_INFINITY;
    printFastint(x);
    x = +x;
    printFastint(x);
    x = -1 / Number.POSITIVE_INFINITY;
    printFastint(x);
    x = +x;
    printFastint(x);

    for (i = 0; i < vals.length; i++) {
        x = vals[i];
        x += 0.5;  // force non-fastint
        x -= 0.5;  // stays non-fastint
        y = +x;    // force fastint
        printFastint(x);
        printFastint(y);
    }
}

try {
    binaryArithmeticTest();
    binaryArithmeticBrute();
    binaryBitopsBrute();
    unaryArithmeticTest();
    unaryArithmeticBrute();
    unaryBitopsBrute();
    negativeZeroTest();
    retvalDowngradeTest();
    yieldResumeDowngradeTest();
    unaryPlusDowngradeTest();
    downgradeSanityTest();
} catch (e) {
    print(e.stack || e);
}
